// Keen Kira
// for the LEDmePlay
//
// www.mithotronic.de
//
// Version: 1.0.0
// Authors: Michael Rosskopf (2023)
//
// Thanks to Thomas Laubach!
//
// Release Notes:
// V1.0.0: First release
// V1.0.1: Little correction of parrot movement

// Include libraries for adafruit matrix
#include <Adafruit_GFX.h>   // Core graphics library
#include <RGBmatrixPanel.h> // Hardware-specific library
#include <avr/pgmspace.h> // Necessary in order to maintain the maze data in program memory

// Setup adafruit matrix
#define CLK 50
#define OE  51
#define LAT 10
#define A   A0
#define B   A1
#define C   A2
#define D   A3
RGBmatrixPanel matrix(A, B, C, D, CLK, LAT, OE, false);

// Audio out
int audio = 2;

// Define notes
#define NOTE_B0  31
#define NOTE_C1  33
#define NOTE_CS1 35
#define NOTE_D1  37
#define NOTE_DS1 39
#define NOTE_E1  41
#define NOTE_F1  44
#define NOTE_FS1 46
#define NOTE_G1  49
#define NOTE_GS1 52
#define NOTE_A1  55
#define NOTE_AS1 58
#define NOTE_B1  62
#define NOTE_C2  65
#define NOTE_CS2 69
#define NOTE_D2  73
#define NOTE_DS2 78
#define NOTE_E2  82
#define NOTE_F2  87
#define NOTE_FS2 93
#define NOTE_G2  98
#define NOTE_GS2 104
#define NOTE_A2  110
#define NOTE_AS2 117
#define NOTE_B2  123
#define NOTE_C3  131
#define NOTE_CS3 139
#define NOTE_D3  147
#define NOTE_DS3 156
#define NOTE_E3  165
#define NOTE_F3  175
#define NOTE_FS3 185
#define NOTE_G3  196
#define NOTE_GS3 208
#define NOTE_A3  220
#define NOTE_AS3 233
#define NOTE_B3  247
#define NOTE_C4  262
#define NOTE_CS4 277
#define NOTE_D4  294
#define NOTE_DS4 311
#define NOTE_E4  330
#define NOTE_F4  349
#define NOTE_FS4 370
#define NOTE_G4  392
#define NOTE_GS4 415
#define NOTE_A4  440
#define NOTE_AS4 466
#define NOTE_B4  494
#define NOTE_C5  523
#define NOTE_CS5 554
#define NOTE_D5  587
#define NOTE_DS5 622
#define NOTE_E5  659
#define NOTE_F5  698
#define NOTE_FS5 740
#define NOTE_G5  784
#define NOTE_GS5 831
#define NOTE_A5  880
#define NOTE_AS5 932
#define NOTE_B5  988
#define NOTE_C6  1047
#define NOTE_CS6 1109
#define NOTE_D6  1175
#define NOTE_DS6 1245
#define NOTE_E6  1319
#define NOTE_F6  1397
#define NOTE_FS6 1480
#define NOTE_G6  1568
#define NOTE_GS6 1661
#define NOTE_A6  1760
#define NOTE_AS6 1865
#define NOTE_B6  1976
#define NOTE_C7  2093
#define NOTE_CS7 2217
#define NOTE_D7  2349
#define NOTE_DS7 2489
#define NOTE_E7  2637
#define NOTE_F7  2794
#define NOTE_FS7 2960
#define NOTE_G7  3136
#define NOTE_GS7 3322
#define NOTE_A7  3520
#define NOTE_AS7 3729
#define NOTE_B7  3951
#define NOTE_C8  4186
#define NOTE_CS8 4435
#define NOTE_D8  4699
#define NOTE_DS8 4978

// Joystick 1
int buttonU1 = 30;
int buttonD1 = 32;
int buttonL1 = 34;
int buttonR1 = 36;
int buttonFire1 = 38;
int analogX1 = 8;
int analogY1 = 9;

// Joystick 2
int buttonU2 = 31;
int buttonD2 = 33;
int buttonL2 = 35;
int buttonR2 = 37;
int buttonFire2 = 39;
int analogX2 = 10;
int analogY2 = 11;

// Sensitivity of analog thumb joysticks (of the LEDmePlay Joypad) in case of "digital usage" (detects movement if deviates from center position value of 512 by sensitivity value)
const int sensitivity = 192;

// Other buttons
int buttonReset = 42;
int buttonPause = 43;

// Define maps
//
// 1st value: Number of tiles in x-direction
// 2nd value: Number of tiles in y-direction
// 3rd value: Level color: 0 == Invisible, 1 == Blue, 2 == Green, 3 == Turquoise, 4 == Red, 5 == Violet, 6 == Yellow, 7 == White
//
//   type 0     type 1     type 2     type 3     type 4     type 5     type 6     type 7     type 8
//   --------   --------   --------   O-------   -------O   O-------   -------O   O-------   -------O
//   --------   --------   --------   O-------   -------O   O-------   -------O   O-------   -------O
//   --------   --------   --------   O-------   -------O   O-------   -------O   O-------   -------O
//   --------   --------   --------   O-------   -------O   O-------   -------O   O-------   -------O
//   --------   --------   --------   O-------   -------O   O-------   -------O   O-------   -------O
//   --------   --------   --------   O-------   -------O   O-------   -------O   O-------   -------O
//   --------   --------   -O-O-O-O   O-------   -------O   O-------   -------O   OO-O-O-O   -O-O-O-O
//   --------   OOOOOOOO   O-O-O-O-   O-------   -------O   OOOOOOOO   OOOOOOOO   O-O-O-O-   O-O-O-OO
//
// The remaining numbers encode the tiles, flowers, and enemies.
// Add tile type value (0 - 8)
// to values of
// Normal flower:            10
// Extralife flower:         20
// Cloud:                    30
// Fireball flying left:     40
// Fireball flying right:    50
// Big Bird:                 60
// Parrot:                   70
// Swirl:                    80
// Humming top:              90
// Devil:                   100
// Comet                    110
// Hornet:                  120
// Spider:                  130
// Spider upside down:      140
// Laser:                   150
// Vertical moving laser    160
// Horizontal moving laser  170
// Vertical barrier:        180
// Horizontal barrier:      190
//
// Play start position      240
//
// Example: Tile 1 with Parrot (value 70) is encoded by 71.
//
const byte numberOfLevels = 16;
const uint8_t levels[] PROGMEM  = {                                         
                                         // Level 1: Green field
                                         8,   8,   2,
                                         0,   0,   0,  30,   0,   0,   0,   0,
                                        11,   0,   0,   0,   0,   0,   0,  11,
                                         0,   0,   0,   0,   0,   0,   0,   0,
                                         0,  11,   0,   0,   0,   0,  11,   0,
                                         0,   0,  11,   0,   0,  11,   0,   0,
                                         0,   0,   0,  21,   1,   0,   0,  40,
                                        50,   0,   0,   0,   0,   0,   0,   0,
                                       242,  12,   2,   2,   2,   2,  12,   2,

                                         // Level 2: Birds 
                                        12,   8,   3,
                                         0,   0,   0,  60,  30,   0,   0,  30,   0,   0,   0,   0,
                                         0, 241,  11,   0,   0,   0,   0,   0,   0,   0,   1,  21,
                                         0,   5,   6,   0,   0,   0,  60,   1,  11,   0,   5,   6,
                                         0,   0,   0,   0,   1,  11,   0,   5,   6,   0,   0,   0,
                                         0,   1,  11,   0,   5,   6,   0,   0,   0,  60,   1,  11,
                                         0,   5,   6,   0,   0,   0,   0,   1,  11,   0,   5,   6,
                                        60,   0,   0,   0,   0,   0,   0,   5,   6,   0,   0,   0,
                                         2,   2,  12,   2,   2,  12,   2,   2,   2,   2,   2,  12,

                                         // Level 3: Planet surface
                                        24,   4,   3,
                                         0,   0, 110,   0,   0,   0, 110,  12,   2,   0,   0, 110,   0,   0,   0, 110,   0,   3,   4,  60,   0,   5,   6,   0, 
                                         0,   0,   0,   2,  12,   0,   0,   3,   4,   0,   0,   0,   0,   0,   0,   0,   0,   5,   6,   0,   0,   1,   1,  60, 
                                         0,   0,   0,   3,   4,   0,   0,   3,   4,   0,   0,   2,  12, 112,  12,   0,   0,   1,   1,   0,  60,   3,   4,   0, 
                                       242,  12,   2,   3,   4,   2,   2,   3,   4,   2,   2,   3,   0,   0,   4,   2,   2,   3,   4,  12,   2,   3,   4,  12,

                                         // Level 4: The Spiral
                                        12,  12,   1,
                                        60,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,  40,
                                         0,   0,   1,   1,   1,  41,  11,   1,   1,   1,   1,  11,
                                         0,   0,   3,   0,   0,   0,   0,   0,   0,   0,  60,   0,
                                         0,   0,   3,   0,   1,   1,  11,   1,   1,  71,   0,   0,
                                         0,   0,   3,   0,   3,   0,   0,   0,   0,  26,   0,   0,
                                         0,   0,   3,   0,   3,   0,   0,   1,   0,   4,   0,   0,
                                         0,  60,   3,   0,   3,   0,   0,   4,   0,   4,   0,   0,
                                         0,   0,   3,   0,   5, 241,  11,   6,   0,   4,   0,  60,
                                         0,   0,   3,   0,   0,   0,   0,   0,   0,   4,   0,   0,
                                         0,   0,   5,  71,   1,   1,  11,   1,   1,   6,   0,   0,
                                        50,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
                                        12,   2,   2,   2,   2,  52,  12,   2,   2,   2,   2,   2,

                                        // Level 5: We fly to fairyland
                                        16,   8,   2,
                                         0,   0,   0,   0,   0,   0,   3,   4,   0,   0,   0,   0,   0,   0,   0,   0,
                                       241,   1,   1,   0,   0,   0,   3,   4,   0,   0, 131,  11,   1,   1,   0,   0,
                                         0,   0,   0,   0,   0,  70,   3,   4,   0,   0,   0,   0,   0, 140,   0,   0,
                                         0,   1,   1,   1,  11,   1,   3,   4,   0,   1,  11,   1,   1,  70,   1,  11,
                                         0,   0, 140,   0,   0,   0,   3,   4,   0,   0,   0, 140,   0,   0,   0,   0,
                                         1,   1,   1,  11, 131,   0,   5,   6,   0,   0,   1,   1,   1, 131,   0,   0,
                                       140,   0,   1,  11,   6,  70,   0,   0,   0,   0,  25,   1,  70,   0,   0,   0,
                                         2,   2,   2,   2,   2,   2,   2,  12,   2,   2,  17,   2,   2,   2,   2,   2,
                                  
                                         // Level 6: Catch me if you can 
                                         8,  16,   6,
                                         0,   0,   0,  60,   0,   0,   0, 100,
                                         0,   1,  11,   0,   0,  21,   1,   0,
                                         0,   3,   0,   0,   0,   0,   4,   0,
                                         0,   3,  11,   0,   0,  11,   4,   0,
                                         0,   3,   0,   0,   0,   0,   4,   0,
                                        50,   0,   0,   0,   0,   0,   0,  40,
                                         0,   3,   0,   0,   0,   0,   4,   0,
                                         0,   3,  11,   0,   0,  11,   4,   0,
                                         0,   3,   0,   0,  60,   0,   4,   0,
                                        50,   0,   0,   0,   0,   0,   0,  40,
                                         0,   3,   0,   0,   0,   0,   4,   0,
                                         0,   3,  11,   0,   0,  11,   4,   0,
                                         0,   3,   0,   0,   0,   0,   4,   0,
                                         0,   5,  11,   0,   0,  11,   6,   0,
                                         0,   0,   0,   0,   0,   0,   0,   0,
                                       242,   2,   2,   2,   2,   2,   2,   2,

                                        // Level 7: We will block you
                                        12,   8,   4,
                                       241,   1,   1,   1,   1,   1,   1,   1,   0,   1,   1,  11,
                                         0,   1,   1,   1,   1,   1,   1, 191,  11,   1,   1,   1,
                                         0,   3, 190,   0,   1,   1,   1,   1,   1, 151,   0, 180,
                                         1,   1,  11,   6,   0,   0,  21,   6, 180,   4,   0,   0,
                                         0,   0,   0,   0,   0,  11, 131,   6,   0,   4,   1,   0,
                                        11,   0,   1,   1,  11,   1, 131,   6,   0,   0,  16,   0,
                                         0,   0, 140,   0,   0,   0,   0,   0,   0,   0,   0,   0,
                                       131,   1,   1,  11,   1,   1,   1,   1,  11,   1,   1,   1,                                      
                                         
                                        // Level 8: Humming tops are coming
                                        12,   8,   2,
                                         0,   0,   0,   0,  11,   0,   0,  11,   0,  90,   0,   0,
                                        11,   0,   0,   0,   0,   0,   0,   0,  90,   0,   0,  11,
                                         0,   0,   0,   0,   0,  11,  21,  90,   0,   0,   0,   0,
                                         0,   0,  11,   0,   0,   0,  90,   0,   0,  11,   0,   0,
                                        11,   0,   0,   0,   0,  90,   0,   0,   0,   0,   0,  11,
                                         0,   0,  11,   0,  90,   0,   0,   0,   0,  11,   0,   0,
                                         0,   0,   0,  90,  11,   0,   0,  11,   0,   0,   0,   0,
                                       242,   2,  90,   0,   0,   0,   0,   0,   0,   0,   0,   2,
                                         
                                         // Level 9: Caverns
                                        12,  12,   7,
                                         0,   0,   0,   0,   0,   3,   4,   0,   0,   0,   0,   0,
                                       242,   2,   0,  12,   2,   3,   4,   0,   0,   1,  21,   0,
                                         0,   4,   0,   3,   0,   0,   4,   0,   0,   0,   0,   0,
                                         0,   4,   0,   3,   0,   0,   4,   0,   0,   0,   0,   0,
                                         0,   4,   0,   3,   0,   0,   4,   0,   0,   0,   0,   0,
                                         0,   4,   0,   5,   1,   1,   6,  12,   2,  72,   2,   0,
                                         0,   4,   0,   3,   0,  60,   0,   0,  60,   0,   4,   0,
                                         0,   4,   0,   3,   0,   0,   0,   0,   0,   0,   4,   0,
                                         0,   4,   0,   3,   0,  71,  11,  11,   1,   0,   4,   0,
                                         0,   4, 160,   3,   0, 171,   1,   1,   1,   0,   4, 160,
                                         0,   4,   1,  11,   6,   0,   0,   0,   0,   5,  11,   1,
                                         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,

                                         // Level 10: Be fast
                                        16,   8,   4,
                                       241,   0,   0,   0, 110,   0,   0,   0,   0,   0,   0, 110,   0,   0,   0,   0,
                                         0,   0,   0,   0,   0,   0,   0,  11,  11,   0,   0,   0,   0,   0,   0, 120,
                                         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0, 120,   0,
                                         1,   1,   1,   1,   1,   1,   1,  11,  11,   1,   1,   1,   1, 121,   0,   0,
                                        21,   0,   0,   0,   0,   0,   0, 120,   0,   0,   0,   0,   0,   0,   0,   0,
                                         0,   0,   0,  11,  11,   0, 120,   0,   0,   0,   0,  11,  11,   0,   0,   0,
                                         0,   0,   0,   0,   0, 120,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
                                        12,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,   2,
                                         
                                        // Level 11: Outside world
                                        24,   6,   6,
                                         0,   0,   0,   0,   0,   0,   0, 110,   0,  30,   0,   0,   0,   0,   0,   0,   0,   0, 110,  30,   0,   0, 110,   0, 
                                        21,   0,   0,   0,   0,  90,   0,   0,   0,   0,   0,   0,  91,  11,   0,   0,   0,   0,   0,   0,   0,   0,   1,  11, 
                                         0,   0,   0,   0,   0,   0,   0,   0,   0, 110,   0,   0,   0,   0,   0,   0,   0,   0,   0,   2,  12,   0,   0,   0, 
                                         0,   0,   1,  11,  90,   0,   1,  11,   0,   2,  12,   0,   0,  90,   1,  11,   0,   0,   2,   3, 114,   2,   0,   0, 
                                         0,   0,   0,   0,   0,   0,   0,   0,   0,   3,   4, 110,   0,   0,   0,   0,   0,  12,   3,   0,   0,   4,   2,   0, 
                                       242,   2,  12,  90,   0,   0,   2,   2,  12,   3,   4,   2,   0,   0,  90,   2,   2,   3,   0,   0,   0,   0,   4,  12, 

                                        // Level 12: Look up
                                        12,  10,   5,
                                        11,   0,   4, 180,   3,   0,  11,   4, 180,   3,   0,  21,
                                         0,   0,   4,   0,   3,  11,   0,   4,   0,   3,   0,   0,
                                         0,   0,   4,   0,   3,   0,  11,   4,   0,   3,   0,   0,
                                         0,  11,   0,   0,   0,   0,   0,   0,   0,   0,  11, 120,
                                         0,   0,   0,   0,   0,   0,   0,   0,   0,   0, 120,   0,
                                         0,   0,   0,   0,   0,   0,   0,   0,   0, 120,   0,   0,
                                         0,  11,   0,   0,   0,   0,   0,   0, 120,   0,  11,   0,
                                         0,   0,   4,   0,   3, 190,   0,   4,   0,   3,   0,   0,
                                         0,   0,   4,   0,   3,   0,   0,   4,   0,   3,   0,   0,
                                       242,   2,   8,  12,   7,   2,   2,   8,  12,   7,   2,  12,
                                         
                                       // Level 13: Chaos
                                        10,  10,   5,
                                        11,   1,   0,   0,  80, 100,   0,   0,   1,  21,
                                         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
                                         0,   1,  11,   0,   0,   0,   0,   1,  11, 190,
                                         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
                                         0,  11,   1,   0,   0,   0,   0,  11,   1,   0,
                                         0,   5,   6,   0,   1,  11,   0,   5,   6,   0,
                                         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
                                         0,   0,   1,  11,   0,   0,   1,  11,   0, 190,
                                         0,   0,   5,   6,   0,   0,   5,   6,   0,   0,
                                       242,   2,   2,  12,  90,  90,   2,   2,  12,   2,

                                        // Level 14: Staircase of terror
                                        12,  16,   1,
                                        21,   0,   0,   0,  30,   0,   0,  30,   0,   0,   0,   0, 
                                         0,   0,   0,   0,   0,   0,  11,   0,   1,  11,   0,   0, 
                                         0,   0,   0,   0,   0,   1,   0,   0,   0,   0,   0,   0, 
                                         0,   0,  80,   0,   0,   0,   1,   0,   0,  80,   0,   0, 
                                         0,   0,  11,   1,   0,   1,   0,   0,   0,   0,   0,   0, 
                                         0,   0,   0,   0,   0,   0,   1,   0,  11,   1,   0,   0, 
                                         0,   0,   0,   0,   0,   1,   0,   0,   0,   0,   0,   0, 
                                       100,   0,   0,   0,   0,   0,   1,   0,   0,   0,   0, 100, 
                                         0,   0,   1,  11,   0,   1,   0,   0,   0,   0,   0,   0, 
                                         0,   0,   0,   0,   0,   0,   1,   0,   1,  11,   0,   0, 
                                         0,   0,   0,   0,   0,   1,   0,   0,   0,   0,   0,   0, 
                                         0,   0,  80,   0,   0,   0,   1,   0,   0,  80,   0,   0, 
                                         0,   0,  11,   1,   0,   1,   0,   0,   0,   0,   0,   0, 
                                         0,   0,   0,   0,   0,   0,   1,   0,   1,   1,   0,   0, 
                                         0,   0,   0,   0,  30,   1,   0,   0,  30,   0,   0,   0, 
                                       242,   2,   2,   2,   2,   2,   2,  12,   2,   2,  12,   2, 

                                         // Level 15: Boomerangs
                                        12,  12,   7,
                                         0,   0,   0,  80,   0,   0,   0,   0,  80,   0,   0,   0,
                                         0,   1,  11,   1,   0,   0,   1,  11,   1,   0,   0,   0,
                                         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
                                         0,   0,   0,   1,   1,  11,   1,   0,   1,   1,   1,  21,
                                         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
                                         0,   0,   1,  11,   1,   0,   0,   1,  11,   1,   0,   0,
                                         0,   0,   0,  80,   0,   0,   0,   0,  80,   0,   0,   0,
                                        11,   1,   1,   1,   0,   0,  11,   1,   0,   0,   1,  11,
                                         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
                                         0,   0,   0,   1,  11,   1,   0,   0,   1,  11,   1,   0,
                                         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
                                       242,   2,  12,   2,   2,   2,   2,  12,   2,   2,   2,   2,
                                                                                                                                                                                                        
                                        // Level 16: Tom´s thumb
                                        20,  12,   6,
                                         0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,  40,   0,  40,   0,
                                         0,   0,   0,   0,   0,   0,   0,   0,   0,   1,   1,   0,   0,   0,   0,   0,   0,  40,   0,  40,
                                         0,   0,   0,   0,   0,   0,   0,   0,   1,   3,   4,   1,   0,   0,   0,   0,   0,   0,   0,   0,
                                         0,   0,   0,   0,   0,   0,   0,   1,   3,   0,   0,   4,   1,   0,   0,   0,   0,   0,   0,   0,
                                         0,   0,   0,   0,   0,   0,   1,   3,   0, 131,  11,   0,   4,   1,   0,   0,   0,   0,   0,   0,
                                         0,   0,   0,   0,   0,   1,   5,   0,   0,   0,   0, 120,   0,   4,   1,   0,   0,   0,   0,   0,
                                         0,   0,   0,   0,   1,   3,   0,   0,   1,   0,   1,   0,   0,   0,   4,   1,   0,   0,   0,   0,
                                         0,   0,   0,   1,   3,   0,  11,   1,   6,   0,   3,   0,   1,  11,   0,   4,   1,   0,   0,   0,
                                         0,   0,   1,   3,   0,   0,   0,   0,   4,   0,   3,   0,   0,   0,   0,   0,   4,   1,   0,   0,
                                         0,   1,   5,   1,   1, 131,   1,  11,   6, 160,   5,   1, 131,   1,   1,  11,   1,   6,   1,   0,
                                       241,   3,   0,  11,   1,   1,   1,   1,   1,   1,   0, 171,   1,   1,   1,   1,   0,  11,   6,   0,
                                         7,  22,   2,   7,   2,   2,   2,   2,   2,   8,  12,   7,   2,   2,   2,   8,   2,   2, 152,   2
                                  };
                                  
byte levelMap[256]; // Contains the current level after loading from PROGMEM

byte tileNumberX; // Number of tiles in X-direction of the current map
byte tileNumberY; // Number of tiles in Y-direction of the current map
byte mapWidth;    // Number of pixels in X-direction of the current map (tiles * 8)
byte mapHeight;   // Number of pixels in Y-direction of the current map (tiles * 8)

// Level color
byte levelColor;  // Level platform color: 0 == Invisible, 1 == Blue, 2 == Green, 3 == Turquoise, 4 == Red, 5 == Violet, 6 == Yellow, 7 == White
byte red;
byte green;
byte blue;

// Playfield
byte playfield[48][48]; // Visible buffer on screen

// Variables for game music
int jingle1Length = 12;
int jingle1[] ={
      NOTE_C4, 1,
      NOTE_C3, 1,
      NOTE_E4, 1,
      NOTE_E3, 1,
      NOTE_G4, 2,
      NOTE_G3, 2
};
int jingle2Length = 20;
int jingle2[] ={
      NOTE_C4, 1,
      NOTE_D4, 1,
      NOTE_E4, 1,
      NOTE_F4, 1,
      NOTE_G4, 1,
      NOTE_A4, 1,
      NOTE_B4, 1, 
      NOTE_C5, 2, 
      NOTE_G4, 2,
      NOTE_C5, 4
};

// Position of the screen (screenX and screenY define the upper left position of the complete map)
int screenX;
int screenY;
int screenXNew; // New X position after movement
int screenYNew; // New Y position after movement
boolean redraw; // Update of screen if true

// Positions of the upper left tile which is visible on the screen (all others tiles are drawn relative to it to realize the scrolling)
int upperLeftTileX;
int upperLeftTileXOffset; // Number of pixels of upper left tile in X-direction which are not visible (out of screen)
int upperLeftTileY;
int upperLeftTileYOffset; // Number of pixels of upper left tile in Y-direction which are not visible (out of screen)

// Player
byte lives;                 // Remaining lives
byte level;                 // Current level
int flowers;                // Counts the collected flowers
int playerXScreen;          // X position on screen
int playerYScreen;          // Y position on screen
int playerXScreenNew;       // New X position after movement
int playerYScreenNew;       // New Y position after movement
float playerXMap;           // X position on map
float playerYMap;           // Y position on map
float playerXMapNew;        // New X position after movement
float playerYMapNew;        // New Y position after movement
float playerYVector;        // Counts down to define the maximal jump height

byte playerAnimationPhase;  // Animation phase (0 or 1)

#define LEFT  0
#define RIGHT 1
boolean playerDirection;    // 0 == looking right, 1 == looking left
boolean playerDirectionNew; // New direction after movement

float xSpeed;
float ySpeed;
float xStepCounter;
float yStepCounter;
byte flapAnimation;
boolean flapping;
boolean landed;

// X and Y Positions of the body parts after death
float todesanimationX[5];
float todesanimationY[5];

// Enemies
int enemyXScreen[16];    // X position on screen
int enemyYScreen[16];    // Y position on screen
int enemyXScreenNew[16]; // New X position after movement
int enemyYScreenNew[16]; // New Y position after movement
float enemyXMap[16];     // X position on map
float enemyYMap[16];     // Y position on map
byte enemyType[16];      // Enemy type (1 - 16)
// Some enemies move between two points (X1, Y1) and (X2, Y2). The enemyMovement defines the direction (depends on enemy type).
byte enemyMovement[16];
int enemyX1[16];
int enemyY1[16];
int enemyX2[16];
int enemyY2[16];
byte enemyStatus[16]; // 0 == inactive, 1 == active
byte enemyCounter;    // Number of enemies in current level (16 is the maximum)

// Flowers
int flowerXScreen[16];              // X position on screen
int flowerYScreen[16];              // Y position on screen
int flowerXScreenNew[16];           // New X position after movement
int flowerYScreenNew[16];           // New Y position after movement
float flowerXMap[16];               // X position on map
float flowerYMap[16];               // Y position on map
byte flowerStatus[16];              // 0 == inactive, 1 == normal flower, 2 == extralife flower, 3 == extralife collected, 4 == normal flower collected
byte flowerCounter;                 // Number of flowers in current level (16 is the maximum)
byte flowersToFinishLevel;          // Jump to next level if 0
boolean initializeLevelWithFlowers; // true, if a new level is initialized

// Game synchronization
unsigned long engineLoopStartPoint; // Contains millisconds since start of LEDmePlay at begin of each engine loop (every 20ms)

// Some helpers
int i, j, k, x1, y1, x2, y2;
float x, y, z;
boolean changed;       // Helper used in the status screen

byte animationCounter; // Counts up till 255 and starts again with 0 (animations are synchronized to it)
boolean reset;         // true if reset button has been set

byte jingleActive;
byte notePointer;
byte waitingTimeTillNextNote;


// Firework type
struct Bomb
{
  byte bombStatus;
  float xPosition;
  float yPosition;
  float xSpeed;
  float ySpeed;
  float blast;
  int fuse;
  byte numberOfMaroons;
};

// Firework maroon
struct Maroon
{
  byte maroonStatus;
  float xPosition;
  float yPosition;
  float xSpeed;
  float ySpeed;
  byte r;
  byte g;
  byte b;
  int duration;
};

const byte numberOfBombs = 8;
const byte numberOfMaroons = 64;
Bomb bomb[numberOfBombs];
Maroon maroon[numberOfMaroons];
int nextBomb;
int nextMaroon;
byte explosionSound;

// Function declarations
void setup();
boolean joy1Up();
boolean joy1Down();
boolean joy1Left();
boolean joy1Right();
float joy1XValue();
float joy1YValue();
boolean joy1Fire();
boolean joy1FireA();
boolean joy1FireB();
boolean joy1IsLEDmePlayJoypad();
boolean joy2Up();
boolean joy2Down();
boolean joy2Left();
boolean joy2Right();
float joy2XValue();
float joy2YValue();
boolean joy2Fire();
boolean joy2FireA();
boolean joy2FireB();
boolean joy2IsLEDmePlayJoypad();
void drawM(int x, int y);
void drawT(int x, int y);
void mithotronic();
void setLEDMePlayColor(int i);
void ledMePlay();
void title();
void setupGame();
void setupLevel();
void updateLevelColors();
void drawScreen();
void drawTile(byte _type, int _x, int _y);
void clearTile(byte _type, int _x, int _y);
void drawEnemies(byte i);
void moveEnemies();
void initializeEnemyMovement();
void drawFlowers();
void drawPlayer();
boolean directionLeftFree();
boolean directionRightFree();
boolean directionUpFree();
boolean directionDownFree();
void movePlayer();
byte collisionDetection();
boolean checkForLosingLive();
void checkForFlowers();
void showStatus();
void realistischeTodesanimation();
void printScrollText(int scrollOffset, byte y, byte r, byte g, byte b, boolean draw);
void showScrollText();
void startSequence();
void endSequence();
void playJingle();
void initFirework();
void launchBomb();
void moveBombs();
void segmentation(float x, float y, float xSpeed, float ySpeed);
void moveMaroons();
void showFirework();
void loop();

// Program setup  
void setup() 
{
  // Initialize serial connection
  Serial.begin(9600);

  // Initialize random number generator
  randomSeed(analogRead(40));

  // Initialize joysticks and  buttons
  pinMode(buttonL1, INPUT);     
  pinMode(buttonR1, INPUT);     
  pinMode(buttonU1, INPUT);     
  pinMode(buttonD1, INPUT);     
  pinMode(buttonFire1, INPUT);     
  pinMode(buttonL2, INPUT);     
  pinMode(buttonR2, INPUT);     
  pinMode(buttonU2, INPUT);     
  pinMode(buttonD2, INPUT);     
  pinMode(buttonFire2, INPUT);     
  pinMode(buttonReset, INPUT);     
  pinMode(buttonPause, INPUT);    

  // Activate internal pull-up resistors 
  digitalWrite(buttonL1, HIGH);
  digitalWrite(buttonR1, HIGH);
  digitalWrite(buttonU1, HIGH);
  digitalWrite(buttonD1, HIGH);
  digitalWrite(buttonFire1, HIGH);
  digitalWrite(buttonL2, HIGH);
  digitalWrite(buttonR2, HIGH);
  digitalWrite(buttonU2, HIGH);
  digitalWrite(buttonD2, HIGH);
  digitalWrite(buttonFire2, HIGH);
  digitalWrite(buttonReset, HIGH);
  digitalWrite(buttonPause, HIGH);

  // Initialize matrix and define text mode
  matrix.begin();
  matrix.setTextSize(1);
  matrix.setTextWrap(false);

  // Enter the game loop
  loop();
}

// Joystick inquiry (allows to use a classic joystick or a LEDmePlay Joypad without any configuration)
boolean joy1Up()
{
  if((digitalRead(buttonU1) == LOW && digitalRead(buttonD1) != LOW) || (digitalRead(buttonL1) == LOW && digitalRead(buttonR1) == LOW && analogRead(analogY1) > (512 + sensitivity))){ return true; }
  return false;
}
boolean joy1Down()
{
  if((digitalRead(buttonD1) == LOW && digitalRead(buttonU1) != LOW) || (digitalRead(buttonL1) == LOW && digitalRead(buttonR1) == LOW && analogRead(analogY1) < (512 - sensitivity))){ return true; }
  return false;
}
boolean joy1Left()
{
  if((digitalRead(buttonL1) == LOW && digitalRead(buttonR1) != LOW) || (digitalRead(buttonL1) == LOW && digitalRead(buttonR1) == LOW && analogRead(analogX1) > (512 + sensitivity))){ return true; }
  return false;
}
boolean joy1Right()
{
  if((digitalRead(buttonR1) == LOW && digitalRead(buttonL1) != LOW) || (digitalRead(buttonL1) == LOW && digitalRead(buttonR1) == LOW && analogRead(analogX1) < (512 - sensitivity))){ return true; }
  return false;
}
boolean joy1Fire()
{
  if(digitalRead(buttonFire1) == LOW || (digitalRead(buttonU1) == LOW && digitalRead(buttonD1) == LOW)){ return true; }
  return false;
}
boolean joy1FireA()
{
  if(digitalRead(buttonFire1) == LOW){ return true; }
  return false;
}
boolean joy1FireB()
{
  if(digitalRead(buttonU1) == LOW && digitalRead(buttonD1) == LOW){ return true; }
  return false;
}
boolean joy2Up()
{
  if((digitalRead(buttonU2) == LOW && digitalRead(buttonD2) != LOW) || (digitalRead(buttonL2) == LOW && digitalRead(buttonR2) == LOW && analogRead(analogY2) > (512 + sensitivity))){ return true; }
  return false;
}
boolean joy2Down()
{
  if((digitalRead(buttonD2) == LOW && digitalRead(buttonU2) != LOW) || (digitalRead(buttonL2) == LOW && digitalRead(buttonR2) == LOW && analogRead(analogY2) < (512 - sensitivity))){ return true; }
  return false;
}
boolean joy2Left()
{
  if((digitalRead(buttonL2) == LOW && digitalRead(buttonR2) != LOW) || (digitalRead(buttonL2) == LOW && digitalRead(buttonR2) == LOW && analogRead(analogX2) > (512 + sensitivity))){ return true; }
  return false;
}
boolean joy2Right()
{
  if((digitalRead(buttonR2) == LOW && digitalRead(buttonL2) != LOW) || (digitalRead(buttonL2) == LOW && digitalRead(buttonR2) == LOW && analogRead(analogX2) < (512 - sensitivity))){ return true; }
  return false;
}
boolean joy2Fire()
{
  if(digitalRead(buttonFire2) == LOW || (digitalRead(buttonU2) == LOW && digitalRead(buttonD2) == LOW)){ return true; }
  return false;
}
boolean joy2FireA()
{
  if(digitalRead(buttonFire2)){ return true; }
  return false;
}
boolean joy2FireB()
{
  if(digitalRead(buttonU2) == LOW && digitalRead(buttonD2) == LOW){ return true; }
  return false;
}

// Draw the M of the Mithotronic logo
void drawM(int x, int y)
{
  matrix.fillRect(x + 2, y + 2, 6, 1, matrix.Color333(0, 0, 0));
  matrix.fillRect(x, y + 3, 10, 1, matrix.Color333(0, 0, 0));

  matrix.fillRect(x, y + 4, 2, 6, matrix.Color333(3, 3, 3));
  matrix.fillRect(x + 2, y + 3, 2, 2, matrix.Color333(3, 3, 3));
  matrix.fillRect(x + 4, y + 4, 2, 6, matrix.Color333(3, 3, 3));
  matrix.fillRect(x + 6, y + 3, 2, 2, matrix.Color333(3, 3, 3));
  matrix.fillRect(x + 8, y + 4, 2, 6, matrix.Color333(3, 3, 3));
}

// Draw the T of the Mithotronic logo
void drawT(int x, int y)
{
  matrix.fillRect(x, y + 5, 6, 1, matrix.Color333(0, 0, 0));
  matrix.fillRect(x + 2, y + 10, 3, 1, matrix.Color333(0, 0, 0));

  matrix.fillRect(x, y + 3, 6, 2, matrix.Color333(0, 0, 7));
  matrix.fillRect(x + 2, y, 2, 10, matrix.Color333(0, 0, 7));
  matrix.fillRect(x + 4, y + 8, 1, 2, matrix.Color333(0, 0, 7));
}

// Draw the animated Mithotronic logo and play jingle
void mithotronic()
{
  i = -10;
  do
  {
    drawM(7, i);
    drawT(19, 22 - i);
    i++;
    delay(50);
  }
  while(i <= 11);

  // Jingle on start screen
  tone(audio,NOTE_C4,200);
  delay(400+20);
  tone(audio,NOTE_C4,90);
  delay(200-20);
  tone(audio,NOTE_G4,140);
  delay(400+20);
  tone(audio,NOTE_G4,140);
  delay(200-20);
  tone(audio,NOTE_C5,450);
  delay(600);
  tone(audio,NOTE_AS4,140);
  delay(200-20);
  tone(audio,NOTE_A4,130);
  delay(200-10);
  tone(audio,NOTE_F4,120);
  delay(200);
  tone(audio,NOTE_G4,1000);
  delay(3000);
}

// Sets color for the next character to show the LEDmePLay logo
void setLEDMePlayColor(int i)
{
  switch(i % 9)
  {
    case 0:
    matrix.setTextColor(matrix.Color333(5,0,0));
    break;
    case 1:
    matrix.setTextColor(matrix.Color333(5,2,0));
    break;
    case 2:
    matrix.setTextColor(matrix.Color333(2,5,0));
    break;
    case 3:
    matrix.setTextColor(matrix.Color333(0,5,0));
    break;
    case 4:
    matrix.setTextColor(matrix.Color333(0,5,2));
    break;
    case 5:
    matrix.setTextColor(matrix.Color333(0,2,5));
    break;
    case 6:
    matrix.setTextColor(matrix.Color333(0,0,5));
    break;
    case 7:
    matrix.setTextColor(matrix.Color333(2,0,5));
    break;
    case 8:
    matrix.setTextColor(matrix.Color333(5,0,2));
    break;
  }
}

// Draw the LEDmePlay logo
void ledMePlay()
{
  // Clear screen
  matrix.fillScreen(matrix.Color333(0, 0, 0));

  i = 0;
  do
  {
    // Write 'LEDmePlay'
    setLEDMePlayColor(i++);
    matrix.setCursor(7, 5);
    matrix.println("L");
    setLEDMePlayColor(i++);
    matrix.setCursor(13, 5);
    matrix.println("E");
    setLEDMePlayColor(i++);
    matrix.setCursor(19, 5);
    matrix.println("D");

    setLEDMePlayColor(i++);
    matrix.setCursor(10, 11);
    matrix.println("m");
    setLEDMePlayColor(i++);
    matrix.setCursor(16, 11);
    matrix.println("e");

    setLEDMePlayColor(i++);
    matrix.setCursor(4, 19);
    matrix.println("P");
    setLEDMePlayColor(i++);
    matrix.setCursor(10, 19);
    matrix.println("l");
    setLEDMePlayColor(i++);
    matrix.setCursor(16, 19);
    matrix.println("a");
    setLEDMePlayColor(i++);
    matrix.setCursor(22, 19);
    matrix.println("y");
    
    i++;
    if(i > 81)
    {
      i = 0;
    }

    j = 0;
    do
    {
      j++;
      delay(1);
    }
    while(j < 250 && !joy1Fire() && !joy2Fire());
  }
  while(!joy1Fire() && !joy2Fire());
  tone(audio,1024,20);
  delay(200);
  matrix.fillRect(0, 0, 32, 32, matrix.Color333(0,0,0));
}

// Keen Kira title screen
void title()
{
  // Clear screen
  matrix.fillScreen(matrix.Color333(0, 0, 0));

  // Write 'Keen Kira'
  matrix.setTextColor(matrix.Color333(3,0,1));
  matrix.setCursor(5, 1);
  matrix.println("Keen");
  matrix.setCursor(5, 9);
  matrix.println("Kira");

  // Draw green platforms
  red = 0, green = 1, blue = 0;
  drawTile(1, 8, 16);
  drawTile(1, 16, 16);
  drawTile(1, 0, 24);
  drawTile(1, 8, 24);
  drawTile(1, 16, 24);
  drawTile(1, 24, 24);

  // Draw flowers
  matrix.drawPixel(11, 19, matrix.Color333(1, 1, 0));
  matrix.drawPixel(10, 20, matrix.Color333(1, 1, 0));
  matrix.drawPixel(12, 20, matrix.Color333(1, 1, 0));
  matrix.drawPixel(11, 21, matrix.Color333(1, 1, 0));
  matrix.drawPixel(11, 22, matrix.Color333(0, 1, 0));

  matrix.drawPixel(20, 19, matrix.Color333(1, 0, 0));
  matrix.drawPixel(19, 20, matrix.Color333(1, 0, 0));
  matrix.drawPixel(21, 20, matrix.Color333(1, 0, 0));
  matrix.drawPixel(20, 21, matrix.Color333(1, 0, 0));
  matrix.drawPixel(20, 22, matrix.Color333(0, 1, 0));

  matrix.drawPixel(3, 27, matrix.Color333(0, 1, 1));
  matrix.drawPixel(2, 28, matrix.Color333(0, 1, 1));
  matrix.drawPixel(4, 28, matrix.Color333(0, 1, 1));
  matrix.drawPixel(3, 29, matrix.Color333(0, 1, 1));
  matrix.drawPixel(3, 30, matrix.Color333(0, 1, 0));

  matrix.drawPixel(11, 27, matrix.Color333(1, 0, 0));
  matrix.drawPixel(10, 28, matrix.Color333(1, 0, 0));
  matrix.drawPixel(12, 28, matrix.Color333(1, 0, 0));
  matrix.drawPixel(11, 29, matrix.Color333(1, 0, 0));
  matrix.drawPixel(11, 30, matrix.Color333(0, 1, 0));

  matrix.drawPixel(20, 27, matrix.Color333(1, 1, 0));
  matrix.drawPixel(19, 28, matrix.Color333(1, 1, 0));
  matrix.drawPixel(21, 28, matrix.Color333(1, 1, 0));
  matrix.drawPixel(20, 29, matrix.Color333(1, 1, 0));
  matrix.drawPixel(20, 30, matrix.Color333(0, 1, 0));

  matrix.drawPixel(28, 27, matrix.Color333(0, 1, 1));
  matrix.drawPixel(27, 28, matrix.Color333(0, 1, 1));
  matrix.drawPixel(29, 28, matrix.Color333(0, 1, 1));
  matrix.drawPixel(28, 29, matrix.Color333(0, 1, 1));
  matrix.drawPixel(28, 30, matrix.Color333(0, 1, 0));

  byte i = 0;
  do
  {
    matrix.drawPixel(11, 20, matrix.Color333(0, 0, 1));
    if(i % 8 < 2) { matrix.drawPixel(11, 20, matrix.Color333(0, 0, 5)); }

    matrix.drawPixel(20, 20, matrix.Color333(0, 0, 1));
    if(i % 8 < 2) { matrix.drawPixel(20, 20, matrix.Color333(0, 0, 5)); }

    matrix.drawPixel(3, 28, matrix.Color333(0, 0, 1));
    if(i % 8 < 2) { matrix.drawPixel(3, 28, matrix.Color333(0, 0, 5)); }
    
    matrix.drawPixel(11, 28, matrix.Color333(0, 0, 1));
    if(i % 8 < 2) { matrix.drawPixel(11, 28, matrix.Color333(0, 0, 5)); }
    
    matrix.drawPixel(20, 28, matrix.Color333(0, 0, 1));
    if(i % 8 < 2) { matrix.drawPixel(20, 28, matrix.Color333(0, 0, 5)); }
    
    matrix.drawPixel(28, 28, matrix.Color333(0, 0, 1));
    if(i % 8 < 2) { matrix.drawPixel(28, 28, matrix.Color333(0, 0, 5)); }
    
    i++;
    delay(20);
  }
  while(!joy1Fire()); 
  tone(audio,1024,20);
  delay(200);
  matrix.fillRect(0, 0, 32, 32, matrix.Color333(0,0,0));

  randomSeed(analogRead(40));
}

// Setup a new game
void setupGame()
{
  lives = 3;
  level = 1;
  flowers = 0;
  initializeLevelWithFlowers = true;
}

// Setup a new level
void setupLevel()
{
  // If a live has been lost, the level is not initialized with flowers (only on level start).
  if(initializeLevelWithFlowers)
  {
    for(i = 0; i < 16; i++)
    {
      flowerXScreen[i] = 0;
      flowerYScreen[i] = 0;
      flowerXScreenNew[i] = 0;
      flowerYScreenNew[i] = 0;
      flowerXMap[i] = 0;
      flowerYMap[i] = 0;
      flowerStatus[i] = 0;
    }
    flowerCounter = 0;
    flowersToFinishLevel = 0;
  }
  
  enemyCounter = 0;
  
  // Reset enemies
  for(i = 0; i < 16; i++)
  {
    enemyXScreen[i] = 0;
    enemyYScreen[i] = 0;
    enemyXScreenNew[i] = 0;
    enemyYScreenNew[i] = 0;
    enemyXMap[i] = 0;
    enemyYMap[i] = 0;
    enemyType[i] = 0;
    enemyMovement[i] = 0;
    enemyX1[i] = 0;
    enemyY1[i] = 0;
    enemyX2[i] = 0;
    enemyY2[i] = 0;
    enemyStatus[i] = 0;
  }
  
  // Copy level map and enemies from progmem
  k = 0;
  for(i = 0; i < level; i++)
  {
    tileNumberX = byte(pgm_read_byte_near(levels + k));
    tileNumberY = byte(pgm_read_byte_near(levels + 1 + k));
    levelColor = byte(pgm_read_byte_near(levels + 2 + k));
    updateLevelColors();
    
    if(i < level - 1)
    {
      k = k + ((tileNumberX * tileNumberY) + 3);
    }
  }

  mapWidth = tileNumberX * 8;
  mapHeight = tileNumberY * 8;

  for(i = 0; i < (tileNumberX * tileNumberY); i++)
  {
    j = byte(pgm_read_byte_near(levels + 3 + i + k));
    levelMap[i] = j % 10; // Returns tile type
    j = j / 10; // Returns start position, flowers or enemy
    
    // Set starting position of player
    if(j == 24)
    {
      playerXMap = ((i % tileNumberX) * 8) + 2;
      playerYMap = ((i / tileNumberX) * 8) + 3;
      playerXScreen = 15;
      playerYScreen = 14;

      if(playerXMap > 14)
      {
        screenX = playerXMap - 15;
        if(screenX > mapWidth - 32)
        {
          screenX = mapWidth - 32;
          playerXScreen = 32 - (mapWidth - playerXMap);
        }
      }
      else
      {
        screenX = 0;
        playerXScreen = playerXMap;
      }
      
      if(playerYMap > 13)
      {
        screenY = playerYMap - 14;
        if(screenY > mapHeight - 32)
        {
          screenY = mapHeight - 32;
          playerYScreen = 32 - (mapHeight - playerYMap);
        }
      }
      else
      {
        screenY = 0;
        playerYScreen = playerYMap;
      }
      
      playerXMapNew = playerXMap;
      playerYMapNew = playerYMap;
      playerXScreenNew = playerXScreen;
      playerYScreenNew = playerYScreen;
      playerYVector = 0.0;
      playerDirection = RIGHT;
      playerDirectionNew = RIGHT;
      screenXNew = screenX;
      screenYNew = screenY;
      redraw = true;
    }
        
    // Set normal flower
    if(j == 1 && initializeLevelWithFlowers)
    {
      flowerXMap[flowerCounter] = ((i % tileNumberX) * 8) + 2;
      flowerYMap[flowerCounter] = ((i / tileNumberX) * 8) + 3;
      flowerStatus[flowerCounter] = 1;
      flowersToFinishLevel++;
      if(flowerCounter < 15)
      {
        flowerCounter++;
      }
    }

    // Set extralife flower
    if(j == 2 && initializeLevelWithFlowers)
    {
      flowerXMap[flowerCounter] = ((i % tileNumberX) * 8) + 2;
      flowerYMap[flowerCounter] = ((i / tileNumberX) * 8) + 3;
      flowerStatus[flowerCounter] = 2;
      if(flowerCounter < 15)
      {
        flowerCounter++;
      }
    }

    // Set enemy: Cloud
    if(j == 3)
    {
      enemyXMap[enemyCounter] = ((i % tileNumberX) * 8) + 2;
      enemyYMap[enemyCounter] = ((i / tileNumberX) * 8) + 2;
      enemyType[enemyCounter] = 1;
      enemyMovement[enemyCounter] = 1;
      enemyStatus[enemyCounter] = 1;
    }

    // Set enemy: Fireball flying left
    if(j == 4)
    {
      enemyXMap[enemyCounter] = ((i % tileNumberX) * 8) + 2;
      enemyYMap[enemyCounter] = ((i / tileNumberX) * 8) + 3;
      enemyType[enemyCounter] = 2;
      enemyMovement[enemyCounter] = 1;
      enemyStatus[enemyCounter] = 1;
    }

    // Set enemy: Fireball flying right
    if(j == 5)
    {
      enemyXMap[enemyCounter] = ((i % tileNumberX) * 8) + 2;
      enemyYMap[enemyCounter] = ((i / tileNumberX) * 8) + 3;
      enemyType[enemyCounter] = 3;
      enemyMovement[enemyCounter] = 1;
      enemyStatus[enemyCounter] = 1;
    }

    // Set enemy: Big Bird
    if(j == 6)
    {
      enemyXMap[enemyCounter] = ((i % tileNumberX) * 8) + 1;
      enemyYMap[enemyCounter] = ((i / tileNumberX) * 8) + 3;
      enemyType[enemyCounter] = 4;
      enemyMovement[enemyCounter] = 1;
      enemyX1[enemyCounter] = i % tileNumberX;
      enemyY1[enemyCounter] = i / tileNumberX;
      enemyX2[enemyCounter] = i % tileNumberX;
      enemyY2[enemyCounter] = i / tileNumberX;
      enemyStatus[enemyCounter] = 1;
    }
    
    // Set enemy: Parrot
    if(j == 7)
    {
      enemyXMap[enemyCounter] = ((i % tileNumberX) * 8) + 1;
      enemyYMap[enemyCounter] = ((i / tileNumberX) * 8) + 2;
      enemyType[enemyCounter] = 5;
      enemyMovement[enemyCounter] = 1;
      enemyX1[enemyCounter] = i % tileNumberX;
      enemyY1[enemyCounter] = i / tileNumberX;
      enemyX2[enemyCounter] = i % tileNumberX;
      enemyY2[enemyCounter] = i / tileNumberX;
      enemyStatus[enemyCounter] = 1;
    }

    // Set enemy: Swirl
    if(j == 8)
    {
      enemyXMap[enemyCounter] = ((i % tileNumberX) * 8) + 2;
      enemyYMap[enemyCounter] = ((i / tileNumberX) * 8) + 2;
      enemyType[enemyCounter] = 6;
      enemyMovement[enemyCounter] = 1;
      enemyStatus[enemyCounter] = 1;
    }

    // Set enemy: Humming top
    if(j == 9)
    {
      enemyXMap[enemyCounter] = ((i % tileNumberX) * 8) + 3;
      enemyYMap[enemyCounter] = ((i / tileNumberX) * 8) + 3;
      enemyType[enemyCounter] = 7;
      enemyMovement[enemyCounter] = ((i % tileNumberX) * 8) + 3;
      enemyStatus[enemyCounter] = 1;
    }

    // Set enemy: Devil
    if(j == 10)
    {
      enemyXMap[enemyCounter] = ((i % tileNumberX) * 8) + 2;
      enemyYMap[enemyCounter] = ((i / tileNumberX) * 8) + 2;
      enemyType[enemyCounter] = 8;
      enemyMovement[enemyCounter] = 1;
      enemyStatus[enemyCounter] = 1;
    }

    // Set enemy: Comet
    if(j == 11)
    {
      enemyXMap[enemyCounter] = ((i % tileNumberX) * 8) + 3;
      enemyYMap[enemyCounter] = ((i / tileNumberX) * 8);
      enemyType[enemyCounter] = 9;
      enemyMovement[enemyCounter] = ((i % tileNumberX) * 8) + 3;
      enemyStatus[enemyCounter] = 1;
    }

    // Set enemy: Hornet
    if(j == 12)
    {
      enemyXMap[enemyCounter] = ((i % tileNumberX) * 8) + 2;
      enemyYMap[enemyCounter] = ((i / tileNumberX) * 8) + 3;
      enemyType[enemyCounter] = 10;
      enemyMovement[enemyCounter] = 1;
      enemyX1[enemyCounter] = i % tileNumberX;
      enemyY1[enemyCounter] = i / tileNumberX;
      enemyX2[enemyCounter] = i % tileNumberX;
      enemyY2[enemyCounter] = i / tileNumberX;
      enemyStatus[enemyCounter] = 1;
    }
    
    // Set enemy: Spider
    if(j == 13)
    {
      enemyXMap[enemyCounter] = ((i % tileNumberX) * 8) + 1;
      enemyYMap[enemyCounter] = ((i / tileNumberX) * 8) + 4;
      enemyType[enemyCounter] = 11;
      enemyMovement[enemyCounter] = 1;
      enemyX1[enemyCounter] = i % tileNumberX;
      enemyY1[enemyCounter] = i / tileNumberX;
      enemyX2[enemyCounter] = i % tileNumberX;
      enemyY2[enemyCounter] = i / tileNumberX;
      enemyStatus[enemyCounter] = 1;
    }

    // Set enemy: Spider upside down
    if(j == 14)
    {
      enemyXMap[enemyCounter] = ((i % tileNumberX) * 8) + 1;
      enemyYMap[enemyCounter] = ((i / tileNumberX) * 8);
      enemyType[enemyCounter] = 12;
      enemyMovement[enemyCounter] = 1;
      enemyX1[enemyCounter] = i % tileNumberX;
      enemyY1[enemyCounter] = i / tileNumberX;
      enemyX2[enemyCounter] = i % tileNumberX;
      enemyY2[enemyCounter] = i / tileNumberX;
      enemyStatus[enemyCounter] = 1;
    }
            
    // Set enemy: Laser
    if(j == 15)
    {
      enemyXMap[enemyCounter] = ((i % tileNumberX) * 8) + 3;
      enemyYMap[enemyCounter] = ((i / tileNumberX) * 8);
      enemyType[enemyCounter] = 13;
      enemyMovement[enemyCounter] = 1;
      enemyStatus[enemyCounter] = 1;
    }

    // Set enemy: Vertical moving laser
    if(j == 16)
    {
      enemyXMap[enemyCounter] = ((i % tileNumberX) * 8);
      enemyYMap[enemyCounter] = ((i / tileNumberX) * 8);
      enemyType[enemyCounter] = 14;
      enemyMovement[enemyCounter] = 1;
      enemyX1[enemyCounter] = i % tileNumberX;
      enemyY1[enemyCounter] = i / tileNumberX;
      enemyX2[enemyCounter] = i % tileNumberX;
      enemyY2[enemyCounter] = i / tileNumberX;
      enemyStatus[enemyCounter] = 1;
    }
    // Set enemy: Horizontal moving laser

    if(j == 17)
    {
      enemyXMap[enemyCounter] = ((i % tileNumberX) * 8);
      enemyYMap[enemyCounter] = ((i / tileNumberX) * 8);
      enemyType[enemyCounter] = 15;
      enemyMovement[enemyCounter] = 1;
      enemyX1[enemyCounter] = i % tileNumberX;
      enemyY1[enemyCounter] = i / tileNumberX;
      enemyX2[enemyCounter] = i % tileNumberX;
      enemyY2[enemyCounter] = i / tileNumberX;
      enemyStatus[enemyCounter] = 1;
    }

    // Set enemy: Vertical barrier
    if(j == 18)
    {
      enemyXMap[enemyCounter] = ((i % tileNumberX) * 8) + 2;
      enemyYMap[enemyCounter] = ((i / tileNumberX) * 8) + 2;
      enemyType[enemyCounter] = 16;
      enemyMovement[enemyCounter] = 1;
      enemyX1[enemyCounter] = i % tileNumberX;
      enemyY1[enemyCounter] = i / tileNumberX;
      enemyX2[enemyCounter] = i % tileNumberX;
      enemyY2[enemyCounter] = i / tileNumberX;
      enemyStatus[enemyCounter] = 1;
    }

    // Set enemy: Horizontal barrier
    if(j == 19)
    {
      enemyXMap[enemyCounter] = ((i % tileNumberX) * 8) + 2;
      enemyYMap[enemyCounter] = ((i / tileNumberX) * 8) + 2;
      enemyType[enemyCounter] = 17;
      enemyMovement[enemyCounter] = 1;
      enemyX1[enemyCounter] = i % tileNumberX;
      enemyY1[enemyCounter] = i / tileNumberX;
      enemyX2[enemyCounter] = i % tileNumberX;
      enemyY2[enemyCounter] = i / tileNumberX;
      enemyStatus[enemyCounter] = 1;
    }

    if(j > 2 && enemyCounter < 15)
    {
      enemyCounter++;
    }
  }

  // Initialize enemy movement
  initializeEnemyMovement();  

  // Clear screen
  matrix.fillScreen(matrix.Color333(0,0,0));
  
  // Initialize playfield
  for(i = 0; i < 48; i++)
  {
    for(j = 0; j < 48; j++)
    {
      playfield[i][j] = 0;
    }
  }

  xSpeed = 0.0;
  ySpeed = 0.0;
  xStepCounter = 0.0;
  yStepCounter = 0.0;
  flapAnimation = 0;
  flapping = false;
  landed = false;
  
  animationCounter = 0;

  jingleActive = 0;
  notePointer = 0;
  waitingTimeTillNextNote = 0;  
}

// Set RGB-values depending on level color
// 0 == Invisible
// 1 == Blue
// 2 == Green
// 3 == Turquoise
// 4 == Red
// 5 == Violet
// 6 == Yellow
// 7 == White
void updateLevelColors()
{
  switch(levelColor)
  {
    case 0:
    red = 0;
    green = 0;
    blue = 0;
    break;

    case 1:
    red = 0;
    green = 0;
    blue = 1;
    break;

    case 2:
    red = 0;
    green = 1;
    blue = 0;
    break;

    case 3:
    red = 0;
    green = 1;
    blue = 1;
    break;

    case 4:
    red = 1;
    green = 0;
    blue = 0;
    break;

    case 5:
    red = 1;
    green = 0;
    blue = 1;
    break;

    case 6:
    red = 1;
    green = 1;
    blue = 0;
    break;

    case 7:
    red = 1;
    green = 1;
    blue = 1;
    break;
  }
}

// Draw the screen according to the current x and y position
void drawScreen()
{
  if(redraw)
  {
    // Clear all tiles at old positions
    upperLeftTileX = screenX / 8;
    upperLeftTileXOffset = screenX % 8;
    upperLeftTileY = screenY / 8;
    upperLeftTileYOffset = screenY % 8;

    for(j = 0; j < 5; j++)
    {
      for(i = 0; i < 5; i++)
      {
        clearTile(levelMap[((upperLeftTileY + j ) * tileNumberX) + upperLeftTileX + i], (i * 8) - upperLeftTileXOffset, (j * 8) - upperLeftTileYOffset);
      }
    }
    screenX = screenXNew;
    screenY = screenYNew;
  }
  // Draw all tiles at new positions
  upperLeftTileX = screenX / 8;
  upperLeftTileXOffset = screenX % 8;
  upperLeftTileY = screenY / 8;
  upperLeftTileYOffset = screenY % 8;
    
  for(j = 0; j < 5; j++)
  {
    for(i = 0; i < 5; i++)
    {
      drawTile(levelMap[((upperLeftTileY + j ) * tileNumberX) + upperLeftTileX + i], (i * 8) - upperLeftTileXOffset, (j * 8) - upperLeftTileYOffset);
    }
  }
  redraw = false;
}

// Draws a tile (8*8) at position x, y
//
//   type 0     type 1     type 2     type 3     type 4     type 5     type 6     type 7     type 8
//   --------   --------   --------   O-------   -------O   O-------   -------O   O-------   -------O
//   --------   --------   --------   O-------   -------O   O-------   -------O   O-------   -------O
//   --------   --------   --------   O-------   -------O   O-------   -------O   O-------   -------O
//   --------   --------   --------   O-------   -------O   O-------   -------O   O-------   -------O
//   --------   --------   --------   O-------   -------O   O-------   -------O   O-------   -------O
//   --------   --------   --------   O-------   -------O   O-------   -------O   O-------   -------O
//   --------   --------   -O-O-O-O   O-------   -------O   O-------   -------O   OO-O-O-O   -O-O-O-O
//   --------   OOOOOOOO   O-O-O-O-   O-------   -------O   OOOOOOOO   OOOOOOOO   O-O-O-O-   O-O-O-OO
//
void drawTile(byte _type, int _x, int _y)
{
  if(_type == 0)
  {
  }
  if(_type == 1)
  {
    matrix.drawPixel(_x, _y + 7, matrix.Color333(red * 2, green * 2, blue * 2));
    matrix.drawPixel(_x + 1, _y + 7, matrix.Color333(red * 2, green * 2, blue * 2));
    matrix.drawPixel(_x + 2, _y + 7, matrix.Color333(red * 2, green * 2, blue * 2));
    matrix.drawPixel(_x + 3, _y + 7, matrix.Color333(red * 2, green * 2, blue * 2));
    matrix.drawPixel(_x + 4, _y + 7, matrix.Color333(red, green, blue));
    matrix.drawPixel(_x + 5, _y + 7, matrix.Color333(red, green, blue));
    matrix.drawPixel(_x + 6, _y + 7, matrix.Color333(red, green, blue));
    matrix.drawPixel(_x + 7, _y + 7, matrix.Color333(red, green, blue));
    playfield[_x + 8][_y + 15] = 1;
    playfield[_x + 9][_y + 15] = 1;
    playfield[_x + 10][_y + 15] = 1;
    playfield[_x + 11][_y + 15] = 1;
    playfield[_x + 12][_y + 15] = 1;
    playfield[_x + 13][_y + 15] = 1;
    playfield[_x + 14][_y + 15] = 1;
    playfield[_x + 15][_y + 15] = 1;
  }
  if(_type == 2)
  {
    matrix.drawPixel(_x, _y + 7, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x + 1, _y + 6, matrix.Color333(red * 2, green * 2, blue * 2));
    matrix.drawPixel(_x + 2, _y + 7, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x + 3, _y + 6, matrix.Color333(red * 2, green * 2, blue * 2));
    matrix.drawPixel(_x + 4, _y + 7, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x + 5, _y + 6, matrix.Color333(red, green, blue));
    matrix.drawPixel(_x + 6, _y + 7, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x + 7, _y + 6, matrix.Color333(red, green, blue));
    playfield[_x + 8][_y + 15] = 1;
    playfield[_x + 9][_y + 14] = 1;
    playfield[_x + 10][_y + 15] = 1;
    playfield[_x + 11][_y + 14] = 1;
    playfield[_x + 12][_y + 15] = 1;
    playfield[_x + 13][_y + 14] = 1;
    playfield[_x + 14][_y + 15] = 1;
    playfield[_x + 15][_y + 14] = 1;
  }
  if(_type == 3)
  {
    matrix.drawPixel(_x, _y, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x, _y + 1, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x, _y + 2, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x, _y + 3, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x, _y + 4, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x, _y + 5, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x, _y + 6, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x, _y + 7, matrix.Color333(1, 1, 1));
    playfield[_x + 8][_y + 8] = 1;
    playfield[_x + 8][_y + 9] = 1;
    playfield[_x + 8][_y + 10] = 1;
    playfield[_x + 8][_y + 11] = 1;
    playfield[_x + 8][_y + 12] = 1;
    playfield[_x + 8][_y + 13] = 1;
    playfield[_x + 8][_y + 14] = 1;
    playfield[_x + 8][_y + 15] = 1;
  }
  if(_type == 4)
  {
    matrix.drawPixel(_x + 7, _y, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x + 7, _y + 1, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x + 7, _y + 2, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x + 7, _y + 3, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x + 7, _y + 4, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x + 7, _y + 5, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x + 7, _y + 6, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x + 7, _y + 7, matrix.Color333(1, 1, 1));
    playfield[_x + 15][_y + 8] = 1;
    playfield[_x + 15][_y + 9] = 1;
    playfield[_x + 15][_y + 10] = 1;
    playfield[_x + 15][_y + 11] = 1;
    playfield[_x + 15][_y + 12] = 1;
    playfield[_x + 15][_y + 13] = 1;
    playfield[_x + 15][_y + 14] = 1;
    playfield[_x + 15][_y + 15] = 1;
  }
  if(_type == 5)
  {
    matrix.drawPixel(_x, _y, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x, _y + 1, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x, _y + 2, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x, _y + 3, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x, _y + 4, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x, _y + 5, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x, _y + 6, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x, _y + 7, matrix.Color333(red * 2, green * 2, blue * 2));
    matrix.drawPixel(_x + 1, _y + 7, matrix.Color333(red * 2, green * 2, blue * 2));
    matrix.drawPixel(_x + 2, _y + 7, matrix.Color333(red * 2, green * 2, blue * 2));
    matrix.drawPixel(_x + 3, _y + 7, matrix.Color333(red * 2, green * 2, blue * 2));
    matrix.drawPixel(_x + 4, _y + 7, matrix.Color333(red, green, blue));
    matrix.drawPixel(_x + 5, _y + 7, matrix.Color333(red, green, blue));
    matrix.drawPixel(_x + 6, _y + 7, matrix.Color333(red, green, blue));
    matrix.drawPixel(_x + 7, _y + 7, matrix.Color333(red, green, blue));
    playfield[_x + 8][_y + 8] = 1;
    playfield[_x + 8][_y + 9] = 1;
    playfield[_x + 8][_y + 10] = 1;
    playfield[_x + 8][_y + 11] = 1;
    playfield[_x + 8][_y + 12] = 1;
    playfield[_x + 8][_y + 13] = 1;
    playfield[_x + 8][_y + 14] = 1;
    playfield[_x + 8][_y + 15] = 1;
    playfield[_x + 9][_y + 15] = 1;
    playfield[_x + 10][_y + 15] = 1;
    playfield[_x + 11][_y + 15] = 1;
    playfield[_x + 12][_y + 15] = 1;
    playfield[_x + 13][_y + 15] = 1;
    playfield[_x + 14][_y + 15] = 1;
    playfield[_x + 15][_y + 15] = 1;
  }
  if(_type == 6)
  {
    matrix.drawPixel(_x + 7, _y, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x + 7, _y + 1, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x + 7, _y + 2, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x + 7, _y + 3, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x + 7, _y + 4, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x + 7, _y + 5, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x + 7, _y + 6, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x, _y + 7, matrix.Color333(red * 2, green * 2, blue * 2));
    matrix.drawPixel(_x + 1, _y + 7, matrix.Color333(red * 2, green * 2, blue * 2));
    matrix.drawPixel(_x + 2, _y + 7, matrix.Color333(red * 2, green * 2, blue * 2));
    matrix.drawPixel(_x + 3, _y + 7, matrix.Color333(red * 2, green * 2, blue * 2));
    matrix.drawPixel(_x + 4, _y + 7, matrix.Color333(red, green, blue));
    matrix.drawPixel(_x + 5, _y + 7, matrix.Color333(red, green, blue));
    matrix.drawPixel(_x + 6, _y + 7, matrix.Color333(red, green, blue));
    matrix.drawPixel(_x + 7, _y + 7, matrix.Color333(red, green, blue));
    playfield[_x + 15][_y + 8] = 1;
    playfield[_x + 15][_y + 9] = 1;
    playfield[_x + 15][_y + 10] = 1;
    playfield[_x + 15][_y + 11] = 1;
    playfield[_x + 15][_y + 12] = 1;
    playfield[_x + 15][_y + 13] = 1;
    playfield[_x + 15][_y + 14] = 1;
    playfield[_x + 8][_y + 15] = 1;
    playfield[_x + 9][_y + 15] = 1;
    playfield[_x + 10][_y + 15] = 1;
    playfield[_x + 11][_y + 15] = 1;
    playfield[_x + 12][_y + 15] = 1;
    playfield[_x + 13][_y + 15] = 1;
    playfield[_x + 14][_y + 15] = 1;
    playfield[_x + 15][_y + 15] = 1;
  }
  if(_type == 7)
  {
    matrix.drawPixel(_x, _y, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x, _y + 1, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x, _y + 2, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x, _y + 3, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x, _y + 4, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x, _y + 5, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x, _y + 6, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x, _y + 7, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x + 1, _y + 6, matrix.Color333(red * 2, green * 2, blue * 2));
    matrix.drawPixel(_x + 2, _y + 7, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x + 3, _y + 6, matrix.Color333(red * 2, green * 2, blue * 2));
    matrix.drawPixel(_x + 4, _y + 7, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x + 5, _y + 6, matrix.Color333(red, green, blue));
    matrix.drawPixel(_x + 6, _y + 7, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x + 7, _y + 6, matrix.Color333(red, green, blue));
    playfield[_x + 8][_y + 8] = 1;
    playfield[_x + 8][_y + 9] = 1;
    playfield[_x + 8][_y + 10] = 1;
    playfield[_x + 8][_y + 11] = 1;
    playfield[_x + 8][_y + 12] = 1;
    playfield[_x + 8][_y + 13] = 1;
    playfield[_x + 8][_y + 14] = 1;
    playfield[_x + 8][_y + 15] = 1;
    playfield[_x + 9][_y + 14] = 1;
    playfield[_x + 10][_y + 15] = 1;
    playfield[_x + 11][_y + 14] = 1;
    playfield[_x + 12][_y + 15] = 1;
    playfield[_x + 13][_y + 14] = 1;
    playfield[_x + 14][_y + 15] = 1;
    playfield[_x + 15][_y + 14] = 1;
  }
  if(_type == 8)
  {
    matrix.drawPixel(_x + 7, _y, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x + 7, _y + 1, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x + 7, _y + 2, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x + 7, _y + 3, matrix.Color333(2, 2, 2));
    matrix.drawPixel(_x + 7, _y + 4, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x + 7, _y + 5, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x + 7, _y + 6, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x, _y + 7, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x + 1, _y + 6, matrix.Color333(red * 2, green * 2, blue * 2));
    matrix.drawPixel(_x + 2, _y + 7, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x + 3, _y + 6, matrix.Color333(red * 2, green * 2, blue * 2));
    matrix.drawPixel(_x + 4, _y + 7, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x + 5, _y + 6, matrix.Color333(red, green, blue));
    matrix.drawPixel(_x + 6, _y + 7, matrix.Color333(1, 1, 1));
    matrix.drawPixel(_x + 7, _y + 6, matrix.Color333(red, green, blue));
    playfield[_x + 15][_y + 8] = 1;
    playfield[_x + 15][_y + 9] = 1;
    playfield[_x + 15][_y + 10] = 1;
    playfield[_x + 15][_y + 11] = 1;
    playfield[_x + 15][_y + 12] = 1;
    playfield[_x + 15][_y + 13] = 1;
    playfield[_x + 15][_y + 14] = 1;
    playfield[_x + 8][_y + 15] = 1;
    playfield[_x + 9][_y + 14] = 1;
    playfield[_x + 10][_y + 15] = 1;
    playfield[_x + 11][_y + 14] = 1;
    playfield[_x + 12][_y + 15] = 1;
    playfield[_x + 13][_y + 14] = 1;
    playfield[_x + 14][_y + 15] = 1;
    playfield[_x + 15][_y + 14] = 1;
  }}

// Clear a tile (8*8) at position x, y
void clearTile(byte _type, int _x, int _y)
{
  if(_type == 0)
  {
  }
  if(_type == 1)
  {
    matrix.drawPixel(_x, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 1, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 2, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 3, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 4, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 5, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 6, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 7, matrix.Color333(0, 0, 0));
    playfield[_x + 8][_y + 15] = 0;
    playfield[_x + 9][_y + 15] = 0;
    playfield[_x + 10][_y + 15] = 0;
    playfield[_x + 11][_y + 15] = 0;
    playfield[_x + 12][_y + 15] = 0;
    playfield[_x + 13][_y + 15] = 0;
    playfield[_x + 14][_y + 15] = 0;
    playfield[_x + 15][_y + 15] = 0;
  }
  if(_type == 2)
  {
    matrix.drawPixel(_x, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 1, _y + 6, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 2, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 3, _y + 6, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 4, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 5, _y + 6, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 6, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 6, matrix.Color333(0, 0, 0));
    playfield[_x + 8][_y + 15] = 0;
    playfield[_x + 9][_y + 14] = 0;
    playfield[_x + 10][_y + 15] = 0;
    playfield[_x + 11][_y + 14] = 0;
    playfield[_x + 12][_y + 15] = 0;
    playfield[_x + 13][_y + 14] = 0;
    playfield[_x + 14][_y + 15] = 0;
    playfield[_x + 15][_y + 14] = 0;
  }
  if(_type == 3)
  {
    matrix.drawPixel(_x, _y, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x, _y + 1, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x, _y + 2, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x, _y + 3, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x, _y + 4, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x, _y + 5, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x, _y + 6, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x, _y + 7, matrix.Color333(0, 0, 0));
    playfield[_x + 8][_y + 8] = 0;
    playfield[_x + 8][_y + 9] = 0;
    playfield[_x + 8][_y + 10] = 0;
    playfield[_x + 8][_y + 11] = 0;
    playfield[_x + 8][_y + 12] = 0;
    playfield[_x + 8][_y + 13] = 0;
    playfield[_x + 8][_y + 14] = 0;
    playfield[_x + 8][_y + 15] = 0;
  }
  if(_type == 4)
  {
    matrix.drawPixel(_x + 7, _y, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 1, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 2, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 3, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 4, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 5, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 6, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 7, matrix.Color333(0, 0, 0));
    playfield[_x + 15][_y + 8] = 0;
    playfield[_x + 15][_y + 9] = 0;
    playfield[_x + 15][_y + 10] = 0;
    playfield[_x + 15][_y + 11] = 0;
    playfield[_x + 15][_y + 12] = 0;
    playfield[_x + 15][_y + 13] = 0;
    playfield[_x + 15][_y + 14] = 0;
    playfield[_x + 15][_y + 15] = 0;
  }
  if(_type == 5)
  {
    matrix.drawPixel(_x, _y, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x, _y + 1, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x, _y + 2, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x, _y + 3, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x, _y + 4, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x, _y + 5, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x, _y + 6, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 1, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 2, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 3, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 4, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 5, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 6, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 7, matrix.Color333(0, 0, 0));
    playfield[_x + 8][_y + 8] = 0;
    playfield[_x + 8][_y + 9] = 0;
    playfield[_x + 8][_y + 10] = 0;
    playfield[_x + 8][_y + 11] = 0;
    playfield[_x + 8][_y + 12] = 0;
    playfield[_x + 8][_y + 13] = 0;
    playfield[_x + 8][_y + 14] = 0;
    playfield[_x + 8][_y + 15] = 0;
    playfield[_x + 9][_y + 15] = 0;
    playfield[_x + 10][_y + 15] = 0;
    playfield[_x + 11][_y + 15] = 0;
    playfield[_x + 12][_y + 15] = 0;
    playfield[_x + 13][_y + 15] = 0;
    playfield[_x + 14][_y + 15] = 0;
    playfield[_x + 15][_y + 15] = 0;
  }
  if(_type == 6)
  {
    matrix.drawPixel(_x + 7, _y, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 1, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 2, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 3, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 4, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 5, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 6, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 1, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 2, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 3, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 4, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 5, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 6, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 7, matrix.Color333(0, 0, 0));
    playfield[_x + 15][_y + 8] = 0;
    playfield[_x + 15][_y + 9] = 0;
    playfield[_x + 15][_y + 10] = 0;
    playfield[_x + 15][_y + 11] = 0;
    playfield[_x + 15][_y + 12] = 0;
    playfield[_x + 15][_y + 13] = 0;
    playfield[_x + 15][_y + 14] = 0;
    playfield[_x + 8][_y + 15] = 0;
    playfield[_x + 9][_y + 15] = 0;
    playfield[_x + 10][_y + 15] = 0;
    playfield[_x + 11][_y + 15] = 0;
    playfield[_x + 12][_y + 15] = 0;
    playfield[_x + 13][_y + 15] = 0;
    playfield[_x + 14][_y + 15] = 0;
    playfield[_x + 15][_y + 15] = 0;
  }
  if(_type == 7)
  {
    matrix.drawPixel(_x, _y, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x, _y + 1, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x, _y + 2, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x, _y + 3, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x, _y + 4, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x, _y + 5, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x, _y + 6, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 1, _y + 6, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 2, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 3, _y + 6, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 4, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 5, _y + 6, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 6, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 6, matrix.Color333(0, 0, 0));
    playfield[_x + 8][_y + 8] = 0;
    playfield[_x + 8][_y + 9] = 0;
    playfield[_x + 8][_y + 10] = 0;
    playfield[_x + 8][_y + 11] = 0;
    playfield[_x + 8][_y + 12] = 0;
    playfield[_x + 8][_y + 13] = 0;
    playfield[_x + 8][_y + 14] = 0;
    playfield[_x + 8][_y + 15] = 0;
    playfield[_x + 9][_y + 14] = 0;
    playfield[_x + 10][_y + 15] = 0;
    playfield[_x + 11][_y + 14] = 0;
    playfield[_x + 12][_y + 15] = 0;
    playfield[_x + 13][_y + 14] = 0;
    playfield[_x + 14][_y + 15] = 0;
    playfield[_x + 15][_y + 14] = 0;
  }
  if(_type == 8)
  {
    matrix.drawPixel(_x + 7, _y, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 1, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 2, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 3, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 4, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 5, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 6, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 1, _y + 6, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 2, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 3, _y + 6, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 4, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 5, _y + 6, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 6, _y + 7, matrix.Color333(0, 0, 0));
    matrix.drawPixel(_x + 7, _y + 6, matrix.Color333(0, 0, 0));
    playfield[_x + 15][_y + 8] = 0;
    playfield[_x + 15][_y + 9] = 0;
    playfield[_x + 15][_y + 10] = 0;
    playfield[_x + 15][_y + 11] = 0;
    playfield[_x + 15][_y + 12] = 0;
    playfield[_x + 15][_y + 13] = 0;
    playfield[_x + 15][_y + 14] = 0;
    playfield[_x + 8][_y + 15] = 0;
    playfield[_x + 9][_y + 14] = 0;
    playfield[_x + 10][_y + 15] = 0;
    playfield[_x + 11][_y + 14] = 0;
    playfield[_x + 12][_y + 15] = 0;
    playfield[_x + 13][_y + 14] = 0;
    playfield[_x + 14][_y + 15] = 0;
    playfield[_x + 15][_y + 14] = 0;
  }
}

// Draw enemy i
void drawEnemies(byte i)
{
  x1 = enemyXScreen[i];
  y1 = enemyYScreen[i];
  x2 = enemyXScreenNew[i];
  y2 = enemyYScreenNew[i];

  // Cloud
  if(enemyType[i] == 1)
  {
    // Remove enemy at old position
    if(x1 > -5 && x1 < 32 && y1 > -5 && y1 < 32)
    {   
      matrix.drawPixel(x1 + 1, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 3, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1 + 2, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1 + 2, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1 + 3, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1 + 4, matrix.Color333(0, 0, 0));
      playfield[x1 + 9][y1 + 8] = 0;
      playfield[x1 + 10][y1 + 8] = 0;
      playfield[x1 + 8][y1 + 9] = 0;
      playfield[x1 + 9][y1 + 9] = 0;
      playfield[x1 + 10][y1 + 9] = 0;
      playfield[x1 + 11][y1 + 9] = 0;
      playfield[x1 + 9][y1 + 10] = 0;
      playfield[x1 + 10][y1 + 10] = 0;
      playfield[x1 + 10][y1 + 11] = 0;
      playfield[x1 + 9][y1 + 12] = 0;
    }
  
    // Draw enemy at new position
    if(x2 > -5 && x2 < 32 && y2 > -5 && y2 < 32)
    {   
      matrix.drawPixel(x2 + 1, y2, matrix.Color333(1, 1, 3));
      matrix.drawPixel(x2 + 2, y2, matrix.Color333(1, 1, 1));
      matrix.drawPixel(x2, y2 + 1, matrix.Color333(1, 1, 3));
      matrix.drawPixel(x2 + 1, y2 + 1, matrix.Color333(1, 1, 2));
      matrix.drawPixel(x2 + 2, y2 + 1, matrix.Color333(1, 1, 1));
      matrix.drawPixel(x2 + 3, y2 + 1, matrix.Color333(1, 1, 1));
      matrix.drawPixel(x2 + 1, y2 + 2, matrix.Color333(1, 1, 3));
      matrix.drawPixel(x2 + 2, y2 + 2, matrix.Color333(1, 1, 1));
      playfield[x2 + 9][y2 + 8] = 2;
      playfield[x2 + 10][y2 + 8] = 2;
      playfield[x2 + 8][y2 + 9] = 2;
      playfield[x2 + 9][y2 + 9] = 2;
      playfield[x2 + 10][y2 + 9] = 2;
      playfield[x2 + 11][y2 + 9] = 2;
      playfield[x2 + 9][y2 + 10] = 2;
      playfield[x2 + 10][y2 + 10] = 2;
      if(animationCounter % 128 > 120 && animationCounter % 2 == 0)
      {
        matrix.drawPixel(x2 + 2, y2 + 3, matrix.Color333(7, 7, 7));
        matrix.drawPixel(x2 + 1, y2 + 4, matrix.Color333(7, 7, 7));
        playfield[x2 + 10][y2 + 11] = 2;
        playfield[x2 + 9][y2 + 12] = 2;
        // Sound: Flash
        if(!jingleActive){ tone(audio, 128 / ((animationCounter % 128) - 120) , 25); }
      }
    }  
  }

  // Fireball flying left
  if(enemyType[i] == 2)
  {
    // Remove enemy at old position
    if(x1 > -5 && x1 < 32 && y1 > -5 && y1 < 32)
    {   
      matrix.drawPixel(x1, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 3, y1, matrix.Color333(0, 0, 0));
      playfield[x1 + 8][y1 + 8] = 0;
      playfield[x1 + 9][y1 + 8] = 0;
      playfield[x1 + 10][y1 + 8] = 0;
      playfield[x1 + 11][y1 + 8] = 0;
    }

    // Draw enemy at new position
    if(x2 > -5 && x2 < 32 && y2 > -5 && y2 < 32)
    {
      matrix.drawPixel(x2, y2, matrix.Color333(7, 0, 5));
      if(animationCounter % 2 == 1){ matrix.drawPixel(x2 + 1, y2, matrix.Color333(6, 0, 4)); }
      if(animationCounter % 4 == 3){ matrix.drawPixel(x2 + 2, y2, matrix.Color333(5, 0, 3)); }
      if(animationCounter % 8 == 5){ matrix.drawPixel(x2 + 3, y2, matrix.Color333(4, 0, 2)); }
      playfield[x2 + 8][y2 + 8] = 2;
      playfield[x2 + 9][y2 + 8] = 2;
      playfield[x2 + 10][y2 + 8] = 2;
      playfield[x2 + 11][y2 + 8] = 0;
    }
  }

  // Fireball flying right
  if(enemyType[i] == 3)
  {
    // Remove enemy at old position
    if(x1 > -5 && x1 < 32 && y1 > -5 && y1 < 32)
    {   
      matrix.drawPixel(x1, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 3, y1, matrix.Color333(0, 0, 0));
      playfield[x1 + 8][y1 + 8] = 0;
      playfield[x1 + 9][y1 + 8] = 0;
      playfield[x1 + 10][y1 + 8] = 0;
      playfield[x1 + 11][y1 + 8] = 0;
    }

    // Draw enemy at new position
    if(x2 > -5 && x2 < 32 && y2 > -5 && y2 < 32)
    {
      if(animationCounter % 8 == 5){ matrix.drawPixel(x2, y2, matrix.Color333(4, 0, 2)); }
      if(animationCounter % 4 == 3){ matrix.drawPixel(x2 + 1, y2, matrix.Color333(5, 0, 3)); }
      if(animationCounter % 2 == 1){ matrix.drawPixel(x2 + 2, y2, matrix.Color333(6, 0, 4));
      matrix.drawPixel(x2 + 3, y2, matrix.Color333(7, 0, 5)); }
      playfield[x2 + 8][y2 + 8] = 0;
      playfield[x2 + 9][y2 + 8] = 2;
      playfield[x2 + 10][y2 + 8] = 2;
      playfield[x2 + 11][y2 + 8] = 2;
    }
  }

  // Big Bird
  if(enemyType[i] == 4)
  {
    // Remove enemy at old position
    if(x1 > -5 && x1 < 32 && y1 > -5 && y1 < 32)
    {   
      matrix.drawPixel(x1 + 1, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 4, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 3, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 4, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 5, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1, y1 + 2, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1 + 2, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 4, y1 + 2, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 5, y1 + 2, matrix.Color333(0, 0, 0));
      playfield[x1 + 9][y1 + 8] = 0;
      playfield[x1 + 12][y1 + 8] = 0;
      playfield[x1 + 8][y1 + 9] = 0;
      playfield[x1 + 9][y1 + 9] = 0;
      playfield[x1 + 10][y1 + 9] = 0;
      playfield[x1 + 11][y1 + 9] = 0;
      playfield[x1 + 12][y1 + 9] = 0;
      playfield[x1 + 13][y1 + 9] = 0;
      playfield[x1 + 8][y1 + 10] = 0;
      playfield[x1 + 9][y1 + 10] = 0;
      playfield[x1 + 12][y1 + 10] = 0;
      playfield[x1 + 13][y1 + 10] = 0;
    }

    // Draw enemy at new position
    if(x2 > -5 && x2 < 32 && y2 > -5 && y2 < 32)
    {
      matrix.drawPixel(x2 + 2, y2 + 1, matrix.Color333(int((animationCounter % 32) / 4), 0, 0));
      matrix.drawPixel(x2 + 3, y2 + 1, matrix.Color333(int((animationCounter % 32) / 4), 0, 0));
      playfield[x2 + 10][y2 + 9] = 2;
      playfield[x2 + 11][y2 + 9] = 2;
      if(animationCounter % 16 < 4)
      {
        matrix.drawPixel(x2, y2 + 1, matrix.Color333(2, 1, 0));
        matrix.drawPixel(x2 + 1, y2, matrix.Color333(2, 1, 0));
        matrix.drawPixel(x2 + 4, y2, matrix.Color333(2, 1, 0));
        matrix.drawPixel(x2 + 5, y2 + 1, matrix.Color333(2, 1, 0));
        playfield[x2 + 8][y2 + 9] = 2;
        playfield[x2 + 9][y2 + 8] = 2;
        playfield[x2 + 12][y2 + 8] = 2;        
        playfield[x2 + 13][y2 + 9] = 2;
      }
      else if(animationCounter % 16 < 8 || animationCounter % 16 > 11)
      {
        matrix.drawPixel(x2, y2 + 1, matrix.Color333(2, 1, 0));
        matrix.drawPixel(x2 + 1, y2 + 1, matrix.Color333(2, 1, 0));
        matrix.drawPixel(x2 + 4, y2 + 1, matrix.Color333(2, 1, 0));
        matrix.drawPixel(x2 + 5, y2 + 1, matrix.Color333(2, 1, 0));
        playfield[x2 + 8][y2 + 9] = 2;
        playfield[x2 + 9][y2 + 9] = 2;
        playfield[x2 + 12][y2 + 9] = 2;        
        playfield[x2 + 13][y2 + 9] = 2;        
      }
      else
      {
        matrix.drawPixel(x2, y2 + 2, matrix.Color333(2, 1, 0));
        matrix.drawPixel(x2 + 1, y2 + 1, matrix.Color333(2, 1, 0));
        matrix.drawPixel(x2 + 4, y2 + 1, matrix.Color333(2, 1, 0));
        matrix.drawPixel(x2 + 5, y2 + 2, matrix.Color333(2, 1, 0));
        playfield[x2 + 8][y2 + 10] = 2;
        playfield[x2 + 9][y2 + 9] = 2;
        playfield[x2 + 12][y2 + 9] = 2;        
        playfield[x2 + 13][y2 + 10] = 2;         
      }
    }
  }
  
  // Parrot
  if(enemyType[i] == 5)
  {
    // Remove enemy at old position
    if(x1 > -5 && x1 < 32 && y1 > -5 && y1 < 32)
    {   
      matrix.drawPixel(x1, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 3, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 4, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 5, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 3, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 4, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 5, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1 + 2, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1 + 2, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 3, y1 + 2, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 4, y1 + 2, matrix.Color333(0, 0, 0));
      playfield[x1 + 8][y1 + 8] = 0;
      playfield[x1 + 9][y1 + 8] = 0;
      playfield[x1 + 10][y1 + 8] = 0;
      playfield[x1 + 11][y1 + 8] = 0;
      playfield[x1 + 12][y1 + 8] = 0;
      playfield[x1 + 13][y1 + 8] = 0;
      playfield[x1 + 8][y1 + 9] = 0;
      playfield[x1 + 9][y1 + 9] = 0;
      playfield[x1 + 10][y1 + 9] = 0;
      playfield[x1 + 11][y1 + 9] = 0;
      playfield[x1 + 12][y1 + 9] = 0;
      playfield[x1 + 13][y1 + 9] = 0;
      playfield[x1 + 9][y1 + 10] = 0;
      playfield[x1 + 10][y1 + 10] = 0;
      playfield[x1 + 11][y1 + 10] = 0;
      playfield[x1 + 12][y1 + 10] = 0;
    }

    // Draw enemy at new position
    if(x2 > -5 && x2 < 32 && y2 > -5 && y2 < 32)
    {
      // Moving left
      if(enemyMovement[i] == 0)
      {
        matrix.drawPixel(x2, y2, matrix.Color333(3, 0, 0));
        matrix.drawPixel(x2 + 1, y2, matrix.Color333(1, 0, 0));
        matrix.drawPixel(x2 + 1, y2 + 1, matrix.Color333(0, 1, 0));
        matrix.drawPixel(x2 + 2, y2 + 1, matrix.Color333(0, 1, 0));
        matrix.drawPixel(x2 + 3, y2 + 1, matrix.Color333(0, 2, 0));
        matrix.drawPixel(x2 + 4, y2 + 1, matrix.Color333(0, 2, 0));
        matrix.drawPixel(x2 + 5, y2 + 1, matrix.Color333(0, 3, 0));
        matrix.drawPixel(x2 + 2, y2 + 2, matrix.Color333(0, 1, 0));
        matrix.drawPixel(x2 + 3, y2 + 2, matrix.Color333(0, 1, 0));
        playfield[x2 + 8][y2 + 8] = 2;
        playfield[x2 + 9][y2 + 8] = 2;
        playfield[x2 + 9][y2 + 9] = 2;
        playfield[x2 + 10][y2 + 9] = 2;
        playfield[x2 + 11][y2 + 9] = 2;
        playfield[x2 + 12][y2 + 9] = 2;
        playfield[x2 + 13][y2 + 9] = 2;
        playfield[x2 + 10][y2 + 10] = 2;
        playfield[x2 + 11][y2 + 10] = 2;
        if((animationCounter % 16 / 4) == 0)
        {
          matrix.drawPixel(x2 + 3, y2, matrix.Color333(3, 2, 0));
          matrix.drawPixel(x2 + 4, y2, matrix.Color333(5, 4, 0));          
        }
        else if((animationCounter % 16 / 4) == 2)
        {
          matrix.drawPixel(x2 + 3, y2 + 2, matrix.Color333(3, 2, 0));
          matrix.drawPixel(x2 + 4, y2 + 2, matrix.Color333(5, 4, 0));                    
        }
        else
        {
          matrix.drawPixel(x2 + 3, y2 + 1, matrix.Color333(3, 2, 0));
          matrix.drawPixel(x2 + 4, y2 + 1, matrix.Color333(5, 4, 0));                    
        }
      }
      else
      {
        matrix.drawPixel(x2 + 4, y2, matrix.Color333(3, 0, 0));
        matrix.drawPixel(x2 + 5, y2, matrix.Color333(1, 0, 0));
        matrix.drawPixel(x2, y2 + 1, matrix.Color333(0, 1, 0));
        matrix.drawPixel(x2 + 1, y2 + 1, matrix.Color333(0, 1, 0));
        matrix.drawPixel(x2 + 2, y2 + 1, matrix.Color333(0, 2, 0));
        matrix.drawPixel(x2 + 3, y2 + 1, matrix.Color333(0, 2, 0));
        matrix.drawPixel(x2 + 4, y2 + 1, matrix.Color333(0, 3, 0));
        matrix.drawPixel(x2 + 2, y2 + 2, matrix.Color333(0, 1, 0));
        matrix.drawPixel(x2 + 3, y2 + 2, matrix.Color333(0, 1, 0));
        playfield[x2 + 12][y2 + 8] = 2;
        playfield[x2 + 13][y2 + 8] = 2;
        playfield[x2 + 8][y2 + 9] = 2;
        playfield[x2 + 9][y2 + 9] = 2;
        playfield[x2 + 10][y2 + 9] = 2;
        playfield[x2 + 11][y2 + 9] = 2;
        playfield[x2 + 12][y2 + 9] = 2;
        playfield[x2 + 10][y2 + 10] = 2;
        playfield[x2 + 11][y2 + 10] = 2;
        if((animationCounter % 16 / 4) == 0)
        {
          matrix.drawPixel(x2 + 1, y2, matrix.Color333(3, 2, 0));
          matrix.drawPixel(x2 + 2, y2, matrix.Color333(5, 4, 0));          
        }
        else if((animationCounter % 16 / 4) == 2)
        {
          matrix.drawPixel(x2 + 1, y2 + 2, matrix.Color333(3, 2, 0));
          matrix.drawPixel(x2 + 2, y2 + 2, matrix.Color333(5, 4, 0));                    
        }
        else
        {
          matrix.drawPixel(x2 + 1, y2 + 1, matrix.Color333(3, 2, 0));
          matrix.drawPixel(x2 + 2, y2 + 1, matrix.Color333(5, 4, 0));                    
        }
      }
    }
  }

  // Swirl
  if(enemyType[i] == 6)
  {
    // Remove enemy at old position
    if(x1 > -5 && x1 < 32 && y1 > -5 && y1 < 32)
    {   
      matrix.drawPixel(x1, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1 + 2, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1 + 2, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1, y1 + 2, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1, y1 + 1, matrix.Color333(0, 0, 0));
      playfield[x1 + 8][y1 + 8] = 0;
      playfield[x1 + 9][y1 + 8] = 0;
      playfield[x1 + 10][y1 + 8] = 0;
      playfield[x1 + 10][y1 + 9] = 0;
      playfield[x1 + 10][y1 + 10] = 0;
      playfield[x1 + 9][y1 + 10] = 0;
      playfield[x1 + 8][y1 + 10] = 0;
      playfield[x1 + 8][y1 + 9] = 0;
    }

    // Draw enemy at new position
    if(x2 > -5 && x2 < 32 && y2 > -5 && y2 < 32)
    {
      if(animationCounter % 8 == 0){ matrix.drawPixel(x2, y2, matrix.Color333(7, 7, 7)); }
      if(animationCounter % 8 == 1){ matrix.drawPixel(x2 + 1, y2, matrix.Color333(7, 7, 7)); }
      if(animationCounter % 8 == 2){ matrix.drawPixel(x2 + 2, y2, matrix.Color333(7, 7, 7)); }
      if(animationCounter % 8 == 3){ matrix.drawPixel(x2 + 2, y2 + 1, matrix.Color333(7, 7, 7)); }
      if(animationCounter % 8 == 4){ matrix.drawPixel(x2 + 2, y2 + 2, matrix.Color333(7, 7, 7)); }
      if(animationCounter % 8 == 5){ matrix.drawPixel(x2 + 1, y2 + 2, matrix.Color333(7, 7, 7)); }
      if(animationCounter % 8 == 6){ matrix.drawPixel(x2, y2 + 2, matrix.Color333(7, 7, 7)); }
      if(animationCounter % 8 == 7){ matrix.drawPixel(x2, y2 + 1, matrix.Color333(7, 7, 7)); }
      playfield[x2 + 8][y2 + 8] = 2;
      playfield[x2 + 9][y2 + 8] = 2;
      playfield[x2 + 10][y2 + 8] = 2;
      playfield[x2 + 10][y2 + 9] = 2;
      playfield[x2 + 10][y2 + 10] = 2;
      playfield[x2 + 9][y2 + 10] = 2;
      playfield[x2 + 8][y2 + 10] = 2;
      playfield[x2 + 8][y2 + 9] = 2;
    }
  }

  // Humming top  
  if(enemyType[i] == 7)
  {
    // Remove enemy at old position
    if(x1 > -5 && x1 < 32 && y1 > -5 && y1 < 32)
    {   
      matrix.drawPixel(x1 + 1, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1 + 2, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1 + 2, matrix.Color333(0, 0, 0));
      playfield[x1 + 9][y1 + 8] = 0;
      playfield[x1 + 8][y1 + 9] = 0;
      playfield[x1 + 9][y1 + 9] = 0;
      playfield[x1 + 10][y1 + 9] = 0;
      playfield[x1 + 9][y1 + 10] = 0;
      playfield[x1 + 9][y1 + 11] = 0;
    }

    // Draw enemy at new position
    if(x2 > -5 && x2 < 32 && y2 > -5 && y2 < 32)
    {
      matrix.drawPixel(x2 + 1, y2, matrix.Color333(5, 1, 0));
      matrix.drawPixel(x2 + 1, y2 + 1, matrix.Color333(3, 1, 0));
      if(animationCounter % 8 < 3){ matrix.drawPixel(x2, y2 + 1, matrix.Color333(0, 1, 3)); }
      else if(animationCounter % 8 < 5){ matrix.drawPixel(x2 + 1, y2 + 1, matrix.Color333(0, 2, 5)); }
      else if(animationCounter % 8 < 7){ matrix.drawPixel(x2 + 2, y2 + 1, matrix.Color333(0, 1, 3)); }
      matrix.drawPixel(x2 + 1, y2 + 2, matrix.Color333(3, 1, 0));
      matrix.drawPixel(x2 + 1, y2 + 2, matrix.Color333(2, 1, 0));
      playfield[x2 + 9][y2 + 8] = 2;
      playfield[x2 + 8][y2 + 9] = 2;
      playfield[x2 + 9][y2 + 9] = 2;
      playfield[x2 + 10][y2 + 9] = 2;
      playfield[x2 + 9][y2 + 10] = 2;
      playfield[x2 + 9][y2 + 11] = 2;
    }
  }

  // Devil
  if(enemyType[i] == 8)
  {
    // Remove enemy at old position
    if(x1 > -5 && x1 < 32 && y1 > -5 && y1 < 32)
    {   
      matrix.drawPixel(x1, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 3, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1 + 2, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1 + 2, matrix.Color333(0, 0, 0));
      playfield[x1 + 8][y1 + 8] = 0;
      playfield[x1 + 11][y1 + 8] = 0;
      playfield[x1 + 9][y1 + 9] = 0;
      playfield[x1 + 10][y1 + 9] = 0;
      playfield[x1 + 9][y1 + 10] = 0;
      playfield[x1 + 10][y1 + 10] = 0;
    }

    // Draw enemy at new position
    if(x2 > -5 && x2 < 32 && y2 > -5 && y2 < 32)
    {
      matrix.drawPixel(x2, y2, matrix.Color333(int((animationCounter % 32) / 4), 0, 0));
      matrix.drawPixel(x2 + 3, y2, matrix.Color333(int((animationCounter % 32) / 4), 0, 0));
      matrix.drawPixel(x2 + 1, y2 + 1, matrix.Color333(1, 0, 5));
      matrix.drawPixel(x2 + 2, y2 + 1, matrix.Color333(1, 0, 5));
      matrix.drawPixel(x2 + 1, y2 + 2, matrix.Color333(1, 0, 2));
      matrix.drawPixel(x2 + 2, y2 + 2, matrix.Color333(1, 0, 2));
      playfield[x2 + 8][y2 + 8] = 2;
      playfield[x2 + 11][y2 + 8] = 2;
      playfield[x2 + 9][y2 + 9] = 2;
      playfield[x2 + 10][y2 + 9] = 2;
      playfield[x2 + 9][y2 + 10] = 2;
      playfield[x2 + 10][y2 + 10] = 2;
    }
  }

  // Comet
  if(enemyType[i] == 9)
  {
    // Remove enemy at old position
    if(x1 > -5 && x1 < 32 && y1 > -5 && y1 < 32)
    {   
      matrix.drawPixel(x1 + 1, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1 + 2, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1 + 2, matrix.Color333(0, 0, 0));
      playfield[x1 + 9][y1 + 8] = 0;
      playfield[x1 + 10][y1 + 8] = 0;
      playfield[x1 + 9][y1 + 9] = 0;
      playfield[x1 + 10][y1 + 9] = 0;
      playfield[x1 + 9][y1 + 10] = 0;
      playfield[x1 + 10][y1 + 10] = 0;
    }

    // Draw enemy at new position
    if(x2 > -5 && x2 < 32 && y2 > -5 && y2 < 32)
    {
      if(animationCounter % 8 == 1)
      {
        matrix.drawPixel(x2 + 1, y2, matrix.Color333(1 + random(3), 1 + random(3), 1));
        matrix.drawPixel(x2 + 2, y2, matrix.Color333(1 + random(3), 1 + random(3), 1));
      }
      if(animationCounter % 4 == 1)
      {
        matrix.drawPixel(x2 + 1, y2 + 1, matrix.Color333(1 + random(3), 1 + random(3), 1));
        matrix.drawPixel(x2 + 2, y2 + 1, matrix.Color333(1 + random(3), 1 + random(3), 1));
      }
      if(animationCounter % 2 == 1)
      {
        matrix.drawPixel(x2 + 1, y2 + 2, matrix.Color333(1 + random(3), 1 + random(3), 1));
        matrix.drawPixel(x2 + 2, y2 + 2, matrix.Color333(1 + random(3), 1 + random(3), 1));
      }
      playfield[x2 + 9][y2 + 8] = 2;
      playfield[x2 + 10][y2 + 8] = 2;
      playfield[x2 + 9][y2 + 9] = 2;
      playfield[x2 + 10][y2 + 9] = 2;
      playfield[x2 + 9][y2 + 10] = 2;
      playfield[x2 + 10][y2 + 10] = 2;
    }
  }

  // Hornet
  if(enemyType[i] == 10)
  {
    // Remove enemy at old position
    if(x1 > -5 && x1 < 32 && y1 > -5 && y1 < 32)
    {   
      matrix.drawPixel(x1, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 3, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 3, y1 + 1, matrix.Color333(0, 0, 0));
      playfield[x1 + 8][y1 + 8] = 0;
      playfield[x1 + 11][y1 + 8] = 0;
      playfield[x1 + 8][y1 + 9] = 0;
      playfield[x1 + 9][y1 + 9] = 0;
      playfield[x1 + 10][y1 + 9] = 0;
      playfield[x1 + 11][y1 + 9] = 0;
    }

    // Draw enemy at new position
    if(x2 > -5 && x2 < 32 && y2 > -5 && y2 < 32)
    {
      if(animationCounter % 16 < 8)
      {
        matrix.drawPixel(x2, y2, matrix.Color333(0, 0, 3));
        matrix.drawPixel(x2 + 3, y2, matrix.Color333(0, 0, 3));
        playfield[x2 + 8][y2 + 8] = 2;
        playfield[x2 + 11][y2 + 8] = 2;
      }
      else
      {
        matrix.drawPixel(x2, y2 + 1, matrix.Color333(0, 0, 3));
        matrix.drawPixel(x2 + 3, y2 + 1, matrix.Color333(0, 0, 3));
        playfield[x2 + 8][y2 + 9] = 2;
        playfield[x2 + 11][y2 + 9] = 2;
      }
      matrix.drawPixel(x2 + 1, y2 + 1, matrix.Color333(3, 3, 0));
      matrix.drawPixel(x2 + 2, y2 + 1, matrix.Color333(3, 3, 0));
      playfield[x2 + 9][y2 + 9] = 2;
      playfield[x2 + 10][y2 + 9] = 2;
    }
  }

  // Spider
  if(enemyType[i] == 11)
  {
    // Remove enemy at old position
    if(x1 > -5 && x1 < 32 && y1 > -5 && y1 < 32)
    {   
      matrix.drawPixel(x1 + 2, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 3, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 3, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 4, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1, y1 + 2, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1 + 2, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 4, y1 + 2, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 5, y1 + 2, matrix.Color333(0, 0, 0));
      playfield[x1 + 10][y1 + 8] = 0;
      playfield[x1 + 11][y1 + 8] = 0;
      playfield[x1 + 9][y1 + 9] = 0;
      playfield[x1 + 10][y1 + 9] = 0;
      playfield[x1 + 11][y1 + 9] = 0;
      playfield[x1 + 12][y1 + 9] = 0;
      playfield[x1 + 9][y1 + 10] = 0;
      playfield[x1 + 12][y1 + 10] = 0;
    }

    // Draw enemy at new position
    if(x2 > -5 && x2 < 32 && y2 > -5 && y2 < 32)
    {
      matrix.drawPixel(x2 + 2, y2, matrix.Color333(0, 0, 3));
      matrix.drawPixel(x2 + 3, y2, matrix.Color333(0, 0, 3));
      matrix.drawPixel(x2 + 1, y2 + 1, matrix.Color333(1, 1, 0));
      matrix.drawPixel(x2 + 2, y2 + 1, matrix.Color333(5, 5, 0));
      matrix.drawPixel(x2 + 3, y2 + 1, matrix.Color333(5, 5, 0));
      matrix.drawPixel(x2 + 4, y2 + 1, matrix.Color333(1, 1, 0));
      if(animationCounter % 32 < 24){ matrix.drawPixel(x2, y2 + 2, matrix.Color333(3, 3, 0)); }
      else { matrix.drawPixel(x2 + 1, y2 + 2, matrix.Color333(3, 3, 0)); }
      if(animationCounter % 32 > 8){ matrix.drawPixel(x2 + 5, y2 + 2, matrix.Color333(3, 3, 0)); }
      else { matrix.drawPixel(x2 + 4, y2 + 2, matrix.Color333(3, 3, 0)); }
      playfield[x2 + 10][y2 + 8] = 2;
      playfield[x2 + 11][y2 + 8] = 2;
      playfield[x2 + 9][y2 + 9] = 2;
      playfield[x2 + 10][y2 + 9] = 2;
      playfield[x2 + 11][y2 + 9] = 2;
      playfield[x2 + 12][y2 + 9] = 2;
      playfield[x2 + 9][y2 + 10] = 2;
      playfield[x2 + 12][y2 + 10] = 2;
    }
  }
  
  // Spider upside down
  if(enemyType[i] == 12)
  {
    // Remove enemy at old position
    if(x1 > -5 && x1 < 32 && y1 > -5 && y1 < 32)
    {   
      matrix.drawPixel(x1, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 4, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 5, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 3, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 4, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1 + 2, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 3, y1 + 2, matrix.Color333(0, 0, 0));
      playfield[x1 + 9][y1 + 8] = 0;
      playfield[x1 + 12][y1 + 8] = 0;
      playfield[x1 + 9][y1 + 9] = 0;
      playfield[x1 + 10][y1 + 9] = 0;
      playfield[x1 + 11][y1 + 9] = 0;
      playfield[x1 + 12][y1 + 9] = 0;
      playfield[x1 + 10][y1 + 10] = 0;
      playfield[x1 + 11][y1 + 10] = 0;
    }

    // Draw enemy at new position
    if(x2 > -5 && x2 < 32 && y2 > -5 && y2 < 32)
    {
      if(animationCounter % 32 < 24){ matrix.drawPixel(x2, y2, matrix.Color333(3, 3, 0)); }
      else { matrix.drawPixel(x2 + 1, y2, matrix.Color333(3, 3, 0)); }
      if(animationCounter % 32 > 8){ matrix.drawPixel(x2 + 5, y2, matrix.Color333(3, 3, 0)); }
      else { matrix.drawPixel(x2 + 4, y2, matrix.Color333(3, 3, 0)); }
      matrix.drawPixel(x2 + 1, y2 + 1, matrix.Color333(1, 1, 0));
      matrix.drawPixel(x2 + 2, y2 + 1, matrix.Color333(5, 5, 0));
      matrix.drawPixel(x2 + 3, y2 + 1, matrix.Color333(5, 5, 0));
      matrix.drawPixel(x2 + 4, y2 + 1, matrix.Color333(1, 1, 0));
      matrix.drawPixel(x2 + 2, y2 + 2, matrix.Color333(0, 0, 3));
      matrix.drawPixel(x2 + 3, y2 + 2, matrix.Color333(0, 0, 3));
      playfield[x2 + 9][y2 + 8] = 2;
      playfield[x2 + 12][y2 + 8] = 2;
      playfield[x2 + 9][y2 + 9] = 2;
      playfield[x2 + 10][y2 + 9] = 2;
      playfield[x2 + 11][y2 + 9] = 2;
      playfield[x2 + 12][y2 + 9] = 2;
      playfield[x2 + 10][y2 + 10] = 2;
      playfield[x2 + 11][y2 + 10] = 2;
    }
  }
    
  // Laser
  if(enemyType[i] == 13)
  {
    // Remove enemy at old position
    if(x1 > -8 && x1 < 32 && y1 > -8 && y1 < 32)
    {   
      if(animationCounter % 16 == 1)
      {
        matrix.drawPixel(x1, y1, matrix.Color333(0, 0, 0));
        matrix.drawPixel(x1, y1 + 1, matrix.Color333(0, 0, 0));
        matrix.drawPixel(x1, y1 + 2, matrix.Color333(0, 0, 0));
        matrix.drawPixel(x1, y1 + 3, matrix.Color333(0, 0, 0));
        matrix.drawPixel(x1, y1 + 4, matrix.Color333(0, 0, 0));
        matrix.drawPixel(x1, y1 + 5, matrix.Color333(0, 0, 0));
        matrix.drawPixel(x1, y1 + 6, matrix.Color333(0, 0, 0));
        playfield[x1 + 8][y1 + 8] = 0;
        playfield[x1 + 8][y1 + 9] = 0;
        playfield[x1 + 8][y1 + 10] = 0;
        playfield[x1 + 8][y1 + 11] = 0;
        playfield[x1 + 8][y1 + 12] = 0;
        playfield[x1 + 8][y1 + 13] = 0;
        playfield[x1 + 8][y1 + 14] = 0;
      }
      else if(animationCounter % 16 == 9)
      {
        matrix.drawPixel(x1 + 1, y1, matrix.Color333(0, 0, 0));
        matrix.drawPixel(x1 + 1, y1 + 1, matrix.Color333(0, 0, 0));
        matrix.drawPixel(x1 + 1, y1 + 2, matrix.Color333(0, 0, 0));
        matrix.drawPixel(x1 + 1, y1 + 3, matrix.Color333(0, 0, 0));
        matrix.drawPixel(x1 + 1, y1 + 4, matrix.Color333(0, 0, 0));
        matrix.drawPixel(x1 + 1, y1 + 5, matrix.Color333(0, 0, 0));
        matrix.drawPixel(x1 + 1, y1 + 6, matrix.Color333(0, 0, 0));
        playfield[x1 + 9][y1 + 8] = 0;
        playfield[x1 + 9][y1 + 9] = 0;
        playfield[x1 + 9][y1 + 10] = 0;
        playfield[x1 + 9][y1 + 11] = 0;
        playfield[x1 + 9][y1 + 12] = 0;
        playfield[x1 + 9][y1 + 13] = 0;
        playfield[x1 + 9][y1 + 14] = 0;
      }
    }

    // Draw enemy at new position
    if(x2 > -8 && x2 < 32 && y2 > -8 && y2 < 32)
    {
      if(animationCounter < 64 || (animationCounter > 127 && animationCounter < 192))
      {
        if(animationCounter % 16 == 0)
        {
          matrix.drawPixel(x2, y2, matrix.Color333(1, 1, 7));
          matrix.drawPixel(x2, y2 + 1, matrix.Color333(1, 1, 7));
          matrix.drawPixel(x2, y2 + 2, matrix.Color333(1, 1, 7));
          matrix.drawPixel(x2, y2 + 3, matrix.Color333(1, 1, 7));
          matrix.drawPixel(x2, y2 + 4, matrix.Color333(1, 1, 7));
          matrix.drawPixel(x2, y2 + 5, matrix.Color333(1, 1, 7));
          matrix.drawPixel(x2, y2 + 6, matrix.Color333(1, 1, 7));
          // Sound: Laser on screen
          if(!jingleActive){ tone(audio, 60, 10); }
          playfield[x2 + 8][y2 + 8] = 2;
          playfield[x2 + 8][y2 + 9] = 2;
          playfield[x2 + 8][y2 + 10] = 2;
          playfield[x2 + 8][y2 + 11] = 2;
          playfield[x2 + 8][y2 + 12] = 2;
          playfield[x2 + 8][y2 + 13] = 2;
          playfield[x2 + 8][y2 + 14] = 2;
        }
        else if(animationCounter % 16 == 8)
        {
          matrix.drawPixel(x2 + 1, y2, matrix.Color333(1, 1, 7));
          matrix.drawPixel(x2 + 1, y2 + 1, matrix.Color333(1, 1, 7));
          matrix.drawPixel(x2 + 1, y2 + 2, matrix.Color333(1, 1, 7));
          matrix.drawPixel(x2 + 1, y2 + 3, matrix.Color333(1, 1, 7));
          matrix.drawPixel(x2 + 1, y2 + 4, matrix.Color333(1, 1, 7));
          matrix.drawPixel(x2 + 1, y2 + 5, matrix.Color333(1, 1, 7));
          matrix.drawPixel(x2 + 1, y2 + 6, matrix.Color333(1, 1, 7));
          // Sound: Laser on screen
          if(!jingleActive){ tone(audio, 60, 10); }
          playfield[x2 + 9][y2 + 8] = 2;
          playfield[x2 + 9][y2 + 9] = 2;
          playfield[x2 + 9][y2 + 10] = 2;
          playfield[x2 + 9][y2 + 11] = 2;
          playfield[x2 + 9][y2 + 12] = 2;
          playfield[x2 + 9][y2 + 13] = 2;
          playfield[x2 + 9][y2 + 14] = 2;
        } 
      }
    }
  }

  // Vertical moving laser
  if(enemyType[i] == 14)
  {
    // Remove enemy at old position
    if(x1 > -8 && x1 < 32 && y1 > -8 && y1 < 32)
    {   
      matrix.drawPixel(x1, y1, matrix.Color333(0, 0, 0));
      if(animationCounter < 64 || (animationCounter > 127 && animationCounter < 192))
      {
        if(animationCounter % 8 == 1)
        {
          matrix.drawPixel(x1 + 1, y1, matrix.Color333(0, 0, 0));
          matrix.drawPixel(x1 + 2, y1, matrix.Color333(0, 0, 0));
          matrix.drawPixel(x1 + 3, y1, matrix.Color333(0, 0, 0));
          matrix.drawPixel(x1 + 4, y1, matrix.Color333(0, 0, 0));
          matrix.drawPixel(x1 + 5, y1, matrix.Color333(0, 0, 0));
          matrix.drawPixel(x1 + 6, y1, matrix.Color333(0, 0, 0));
        }
      }
      matrix.drawPixel(x1 + 7, y1, matrix.Color333(0, 0, 0));
      playfield[x1 + 8][y1 + 8] = 0;
      playfield[x1 + 9][y1 + 8] = 0;
      playfield[x1 + 10][y1 + 8] = 0;
      playfield[x1 + 11][y1 + 8] = 0;
      playfield[x1 + 12][y1 + 8] = 0;
      playfield[x1 + 13][y1 + 8] = 0;
      playfield[x1 + 14][y1 + 8] = 0;
      playfield[x1 + 15][y1 + 8] = 0;
    }

    // Draw enemy at new position
    if(x2 > -8 && x2 < 32 && y2 > -8 && y2 < 32)
    {
      matrix.drawPixel(x2, y2, matrix.Color333(0, 0, 3));
      matrix.drawPixel(x2 + 7, y2, matrix.Color333(0, 0, 3));
      playfield[x2 + 8][y2 + 8] = 2;
      playfield[x2 + 15][y2 + 8] = 2;
      if(animationCounter < 64 || (animationCounter > 127 && animationCounter < 192))
      {
        if(animationCounter % 8 == 0)
        {
          matrix.drawPixel(x2 + 1, y2, matrix.Color333(1, 1, 7));
          matrix.drawPixel(x2 + 2, y2, matrix.Color333(1, 1, 7));
          matrix.drawPixel(x2 + 3, y2, matrix.Color333(1, 1, 7));
          matrix.drawPixel(x2 + 4, y2, matrix.Color333(1, 1, 7));
          matrix.drawPixel(x2 + 5, y2, matrix.Color333(1, 1, 7));
          matrix.drawPixel(x2 + 6, y2, matrix.Color333(1, 1, 7));
          // Sound: Laser on screen
          if(!jingleActive){ tone(audio, 60, 10); }
        }
        playfield[x2 + 9][y2 + 8] = 2;
        playfield[x2 + 10][y2 + 8] = 2;
        playfield[x2 + 11][y2 + 8] = 2;
        playfield[x2 + 12][y2 + 8] = 2;
        playfield[x2 + 13][y2 + 8] = 2;
        playfield[x2 + 14][y2 + 8] = 2;
      }
    }
  }
  
  // Horizontal moving laser
  if(enemyType[i] == 15)
  {
    // Remove enemy at old position
    if(x1 > -8 && x1 < 32 && y1 > -8 && y1 < 32)
    {   
      matrix.drawPixel(x1, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1, matrix.Color333(0, 0, 0));
      if(animationCounter < 64 || (animationCounter > 127 && animationCounter < 192))
      {
        if(animationCounter % 8 == 1)
        {
          matrix.drawPixel(x1 + 1, y1 + 1, matrix.Color333(0, 0, 0));
          matrix.drawPixel(x1 + 1, y1 + 2, matrix.Color333(0, 0, 0));
          matrix.drawPixel(x1 + 1, y1 + 3, matrix.Color333(0, 0, 0));
          matrix.drawPixel(x1 + 1, y1 + 4, matrix.Color333(0, 0, 0));
          matrix.drawPixel(x1 + 1, y1 + 5, matrix.Color333(0, 0, 0));
          matrix.drawPixel(x1 + 1, y1 + 6, matrix.Color333(0, 0, 0));
        }
      }
      playfield[x1 + 8][y1 + 8] = 0;
      playfield[x1 + 9][y1 + 8] = 0;
      playfield[x1 + 10][y1 + 8] = 0;
      playfield[x1 + 9][y1 + 9] = 0;
      playfield[x1 + 9][y1 + 10] = 0;
      playfield[x1 + 9][y1 + 11] = 0;
      playfield[x1 + 9][y1 + 12] = 0;
      playfield[x1 + 9][y1 + 13] = 0;
      playfield[x1 + 9][y1 + 14] = 0;
    }

    // Draw enemy at new position
    if(x2 > -8 && x2 < 32 && y2 > -8 && y2 < 32)
    {
      matrix.drawPixel(x2, y2, matrix.Color333(0, 0, 3));
      matrix.drawPixel(x2 + 1, y2, matrix.Color333(0, 0, 3));
      matrix.drawPixel(x2 + 2, y2, matrix.Color333(0, 0, 3));
      playfield[x2 + 8][y2 + 8] = 2;
      playfield[x2 + 9][y2 + 8] = 2;
      playfield[x2 + 10][y2 + 8] = 2;
      if(animationCounter < 64 || (animationCounter > 127 && animationCounter < 192))
      {
        if(animationCounter % 8 == 0)
        {
          matrix.drawPixel(x2 + 1, y2 + 1, matrix.Color333(1, 1, 7));
          matrix.drawPixel(x2 + 1, y2 + 2, matrix.Color333(1, 1, 7));
          matrix.drawPixel(x2 + 1, y2 + 3, matrix.Color333(1, 1, 7));
          matrix.drawPixel(x2 + 1, y2 + 4, matrix.Color333(1, 1, 7));
          matrix.drawPixel(x2 + 1, y2 + 5, matrix.Color333(1, 1, 7));
          matrix.drawPixel(x2 + 1, y2 + 6, matrix.Color333(1, 1, 7));
          // Sound: Laser on screen
          if(!jingleActive){ tone(audio, 60, 10); }
        }
        playfield[x2 + 9][y2 + 9] = 2;
        playfield[x2 + 9][y2 + 10] = 2;
        playfield[x2 + 9][y2 + 11] = 2;
        playfield[x2 + 9][y2 + 12] = 2;
        playfield[x2 + 9][y2 + 13] = 2;
        playfield[x2 + 9][y2 + 14] = 2;
      }
    }
  }
  
  // Vertical barrier
  if(enemyType[i] == 16)
  {
    // Remove enemy at old position
    if(x1 > -5 && x1 < 32 && y1 > -5 && y1 < 32)
    {   
      matrix.drawPixel(x1, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1 + 2, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1, y1 + 3, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1 + 3, matrix.Color333(0, 0, 0));
      playfield[x1 + 8][y1 + 8] = 0;
      playfield[x1 + 10][y1 + 8] = 0;
      playfield[x1 + 9][y1 + 9] = 0;
      playfield[x1 + 9][y1 + 10] = 0;
      playfield[x1 + 8][y1 + 11] = 0;
      playfield[x1 + 10][y1 + 11] = 0;
    }

    // Draw enemy at new position
    if(x2 > -5 && x2 < 32 && y2 > -5 && y2 < 32)
    {
      matrix.drawPixel(x2, y2, matrix.Color333(1, 1, 1));
      matrix.drawPixel(x2 + 2, y2, matrix.Color333(1, 1, 1));
      matrix.drawPixel(x2 + 1, y2 + 1, matrix.Color333(int((animationCounter % 16) / 2), 0, 0));
      matrix.drawPixel(x2 + 1, y2 + 2, matrix.Color333(int((animationCounter % 16) / 2), 0, 0));
      matrix.drawPixel(x2, y2 + 3, matrix.Color333(1, 1, 1));
      matrix.drawPixel(x2 + 2, y2 + 3, matrix.Color333(1, 1, 1));
      playfield[x2 + 8][y2 + 8] = 2;
      playfield[x2 + 10][y2 + 8] = 2;
      playfield[x2 + 9][y2 + 9] = 2;
      playfield[x2 + 9][y2 + 10] = 2;
      playfield[x2 + 8][y2 + 11] = 2;
      playfield[x2 + 10][y2 + 11] = 2;
    }
  }

  // Horizontal barrier
  if(enemyType[i] == 17)
  {
    // Remove enemy at old position
    if(x1 > -5 && x1 < 32 && y1 > -5 && y1 < 32)
    {   
      matrix.drawPixel(x1, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 3, y1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 1, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 2, y1 + 1, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1, y1 + 2, matrix.Color333(0, 0, 0));
      matrix.drawPixel(x1 + 3, y1 + 2, matrix.Color333(0, 0, 0));
      playfield[x1 + 8][y1 + 8] = 0;
      playfield[x1 + 11][y1 + 8] = 0;
      playfield[x1 + 9][y1 + 9] = 0;
      playfield[x1 + 10][y1 + 9] = 0;
      playfield[x1 + 8][y1 + 10] = 0;
      playfield[x1 + 11][y1 + 10] = 0;
    }

    // Draw enemy at new position
    if(x2 > -5 && x2 < 32 && y2 > -5 && y2 < 32)
    {
      matrix.drawPixel(x2, y2, matrix.Color333(1, 1, 1));
      matrix.drawPixel(x2 + 3, y2, matrix.Color333(1, 1, 1));
      matrix.drawPixel(x2 + 1, y2 + 1, matrix.Color333(int((animationCounter % 16) / 2), 0, 0));
      matrix.drawPixel(x2 + 2, y2 + 1, matrix.Color333(int((animationCounter % 16) / 2), 0, 0));
      matrix.drawPixel(x2, y2 + 2, matrix.Color333(1, 1, 1));
      matrix.drawPixel(x2 + 3, y2 + 2, matrix.Color333(1, 1, 1));
      playfield[x2 + 8][y2 + 8] = 2;
      playfield[x2 + 11][y2 + 8] = 2;
      playfield[x2 + 9][y2 + 9] = 2;
      playfield[x2 + 10][y2 + 9] = 2;
      playfield[x2 + 8][y2 + 10] = 2;
      playfield[x2 + 11][y2 + 10] = 2;
    }
  }

  enemyXScreen[i] = x2;
  enemyYScreen[i] = y2;
}

// Move all active enemies
void moveEnemies()
{
  for(i = 0; i < 16; i++)
  {
    if(enemyStatus[i] > 0)
    {
      // Cloud
      if(enemyType[i] == 1)
      {  
        if(enemyMovement[i] == 1)
        {
          if(enemyXMap[i] < (mapWidth - 2) && enemyYMap[i] > 0)
          {
            enemyXMap[i] = enemyXMap[i] + 0.14;
            enemyYMap[i] = enemyYMap[i] - 0.07;
          }
          else
          {
            enemyMovement[i] = byte(random(4)) + 1;
          }
        }
        else if(enemyMovement[i] == 2)
        {
          if(enemyXMap[i] < (mapWidth - 2) && enemyYMap[i] < (mapHeight - 1))
          {
            enemyXMap[i] = enemyXMap[i] + 0.14;
            enemyYMap[i] = enemyYMap[i] + 0.07;
          }
          else
          {
            enemyMovement[i] = byte(random(4)) + 1;
          }
        }
        else if(enemyMovement[i] == 3)
        {
          if(enemyXMap[i] > 0 && enemyYMap[i] < (mapHeight - 1))
          {
            enemyXMap[i] = enemyXMap[i] - 0.14;
            enemyYMap[i] = enemyYMap[i] + 0.07;
          }
          else
          {
            enemyMovement[i] = byte(random(4)) + 1;
          }
        }
        else if(enemyMovement[i] == 4)
        {
          if(enemyXMap[i] > 0 && enemyYMap[i] > 0)
          {
            enemyXMap[i] = enemyXMap[i] - 0.14;
            enemyYMap[i] = enemyYMap[i] - 0.07;
          }
          else
          {
            enemyMovement[i] = byte(random(4)) + 1;
          }
        }
      }
        
      // Fireball flying left
      if(enemyType[i] == 2)
      { 
        if(enemyXMap[i] > -4)
        {
          enemyXMap[i] = enemyXMap[i] - 0.5;
        }
        else
        {
          enemyXMap[i] = mapWidth - 1;
        }      
      }

      // Fireball flying right
      if(enemyType[i] == 3)
      { 
        if(enemyXMap[i] < mapWidth)
        {
          enemyXMap[i] = enemyXMap[i] + 0.5;
        }
        else
        {
          enemyXMap[i] = -3;
        }      
      }
      
      // Big Bird
      if(enemyType[i] == 4)
      { 
        // Moving up
        if(enemyMovement[i] == 0)
        {
          if(enemyYMap[i] > enemyY1[i])
          {
            enemyYMap[i] = enemyYMap[i] - 0.2;
          }
          else
          {
            enemyMovement[i] = 1;
          }
        }
        // Moving down
        else
        {
          if(enemyYMap[i] < enemyY2[i])
          {
            enemyYMap[i] = enemyYMap[i] + 0.2;
          }
          else
          {
            enemyMovement[i] = 0;
          }
        }  
      }    

      // Parrot
      if(enemyType[i] == 5)
      { 
        // Moving left
        if(enemyMovement[i] == 0)
        {
          if(enemyXMap[i] > enemyX1[i])
          {
            enemyXMap[i] = enemyXMap[i] - 0.25;
          }
          else
          {
            enemyMovement[i] = 1;
          }
        }
        else
        {
          if(enemyXMap[i] < enemyX2[i])
          {
            enemyXMap[i] = enemyXMap[i] + 0.25;
          }
          else
          {
            enemyMovement[i] = 0;
          }
        }      
      }

      // Swirl
      if(enemyType[i] == 6)
      {  
        if(enemyMovement[i] == 1)
        {
          if(enemyXMap[i] < (mapWidth - 2) && enemyYMap[i] > 0)
          {
            enemyXMap[i] = enemyXMap[i] + 0.1;
            enemyYMap[i] = enemyYMap[i] - 0.2;
          }
          else
          {
            enemyMovement[i] = byte(random(4)) + 1;
          }
        }
        else if(enemyMovement[i] == 2)
        {
          if(enemyXMap[i] < (mapWidth - 2) && enemyYMap[i] < (mapHeight - 1))
          {
            enemyXMap[i] = enemyXMap[i] + 0.1;
            enemyYMap[i] = enemyYMap[i] + 0.2;
          }
          else
          {
            enemyMovement[i] = byte(random(4)) + 1;
          }
        }
        else if(enemyMovement[i] == 3)
        {
          if(enemyXMap[i] > 0 && enemyYMap[i] < (mapHeight - 1))
          {
            enemyXMap[i] = enemyXMap[i] - 0.1;
            enemyYMap[i] = enemyYMap[i] + 0.2;
          }
          else
          {
            enemyMovement[i] = byte(random(4)) + 1;
          }
        }
        else if(enemyMovement[i] == 4)
        {
          if(enemyXMap[i] > 0 && enemyYMap[i] > 0)
          {
            enemyXMap[i] = enemyXMap[i] - 0.1;
            enemyYMap[i] = enemyYMap[i] - 0.2;
          }
          else
          {
            enemyMovement[i] = byte(random(4)) + 1;
          }
        }
      }

      // Humming top
      if(enemyType[i] == 7)
      {
        if(enemyYMap[i] > -5)
        {
          enemyYMap[i] = enemyYMap[i] - 0.25;
        }
        else
        {
          enemyXMap[i] = enemyMovement[i];
          enemyYMap[i] = mapHeight;
          enemyStatus[i] = random(3) + 1;
        }
        if(enemyStatus[i] == 2)
        {
          enemyXMap[i] = enemyXMap[i] - 0.1;
        }        
        else if(enemyStatus[i] == 3)
        {
          enemyXMap[i] = enemyXMap[i] + 0.1;
        }        
      } 

      // Devil
      if(enemyType[i] == 8)
      {
        // Move (turns towards player´s ship)
        // Calculate directional
        x = (double(playerXMap + 1.0) - (enemyXMap[i] + 2.0));
        y = (double(playerYMap + 2.0) - (enemyYMap[i] + 1.5));
        // Calculate length of directional vector
        z = sqrt(sq(x) + sq(y));
        enemyXMap[i] = enemyXMap[i] + ((x / z) / 20.0);
        enemyYMap[i] = enemyYMap[i] + ((y / z) / 20.0);
        if(enemyXMap[i] < 0)
        {
          enemyXMap[i] = 0;
        }
        if(enemyXMap[i] > (mapWidth - 4))
        {
          enemyXMap[i] = (mapWidth - 4);
        }
        if(enemyYMap[i] < 0)
        {
          enemyYMap[i] = 0;
        }
        if(enemyYMap[i] > (mapHeight - 4))
        {
          enemyYMap[i] = (mapHeight - 4);
        }
      } 

      // Comet
      if(enemyType[i] == 9)
      {
        if(random(100) < 50)
        {
          if(enemyXMap[i] > -1)
          {
            enemyXMap[i] = enemyXMap[i] - double(random(5)) / 20;
          }
        }
        else
        {
          if(enemyXMap[i] < mapWidth - 1)
          {
            enemyXMap[i] = enemyXMap[i] + double(random(5)) / 20;
          }
        }
        if(enemyYMap[i] < mapHeight)
        {
          enemyYMap[i] = enemyYMap[i] + 0.4;
        }        
        else
        {
          enemyYMap[i] = -1;
          enemyXMap[i] = enemyMovement[i];
        }      
      }

      // Hornet
      if(enemyType[i] == 10)
      {
        // Moving left
        if(enemyMovement[i] == 0)
        {
          if(enemyXMap[i] > enemyX1[i])
          {
            enemyXMap[i] = enemyXMap[i] - 0.5;
          }
          else
          {
            enemyMovement[i] = 1;
          }
        }
        else
        {
          if(enemyXMap[i] < enemyX2[i])
          {
            enemyXMap[i] = enemyXMap[i] + 0.5;
          }
          else
          {
            enemyMovement[i] = 0;
          }
        }      
      } 

      // Spider
      if(enemyType[i] == 11)
      {
        // Moving left
        if(enemyMovement[i] == 0)
        {
          if(enemyXMap[i] > enemyX1[i])
          {
            enemyXMap[i] = enemyXMap[i] - 0.2;
          }
          else
          {
            enemyMovement[i] = 1;
          }
        }
        else
        {
          if(enemyXMap[i] < enemyX2[i])
          {
            enemyXMap[i] = enemyXMap[i] + 0.2;
          }
          else
          {
            enemyMovement[i] = 0;
          }
        }      
      } 

      // Spider upside down
      if(enemyType[i] == 12)
      {
        // Moving left
        if(enemyMovement[i] == 0)
        {
          if(enemyXMap[i] > enemyX1[i])
          {
            enemyXMap[i] = enemyXMap[i] - 0.2;
          }
          else
          {
            enemyMovement[i] = 1;
          }
        }
        else
        {
          if(enemyXMap[i] < enemyX2[i])
          {
            enemyXMap[i] = enemyXMap[i] + 0.2;
          }
          else
          {
            enemyMovement[i] = 0;
          }
        }      
      } 

      // Laser
      if(enemyType[i] == 13)
      {
        // Do nothing
      } 

      // Vertical moving laser
      if(enemyType[i] == 14)
      {
        // Moving up
        if(enemyMovement[i] == 0)
        {
          if(enemyYMap[i] > enemyY1[i])
          {
            enemyYMap[i] = enemyYMap[i] - 0.2;
          }
          else
          {
            enemyMovement[i] = 1;
          }
        }
        // Moving down
        else
        {
          if(enemyYMap[i] < enemyY2[i])
          {
            enemyYMap[i] = enemyYMap[i] + 0.2;
          }
          else
          {
            enemyMovement[i] = 0;
          }
        }  
      } 

      // Horizontal moving laser
      if(enemyType[i] == 15)
      {
        // Moving left
        if(enemyMovement[i] == 0)
        {
          if(enemyXMap[i] > enemyX1[i])
          {
            enemyXMap[i] = enemyXMap[i] - 0.2;
          }
          else
          {
            enemyMovement[i] = 1;
          }
        }
        else
        {
          if(enemyXMap[i] < enemyX2[i])
          {
            enemyXMap[i] = enemyXMap[i] + 0.2;
          }
          else
          {
            enemyMovement[i] = 0;
          }
        }      
      } 

      // Vertical barrier
      if(enemyType[i] == 16)
      {
        // Move up
        if(playerYMap < (enemyYMap[i] - 0.1) && enemyYMap[i] > enemyY1[i])
        {
          enemyYMap[i] = enemyYMap[i] - 0.1;
        }
        // Move down
        else if(playerYMap > (enemyYMap[i] + 0.1) && enemyYMap[i] < enemyY2[i])
        {
          enemyYMap[i] = enemyYMap[i] + 0.1;          
        }
      } 

      // Horizontal barrier
      if(enemyType[i] == 17)
      {
        // Move left
        if(playerXMap < (enemyXMap[i] - 1.1) && enemyXMap[i] > enemyX1[i])
        {
          enemyXMap[i] = enemyXMap[i] - 0.1;
        }
        // Move right
        else if(playerXMap > (enemyXMap[i] - 1.9) && enemyXMap[i] < enemyX2[i])
        {
          enemyXMap[i] = enemyXMap[i] + 0.1;          
        }
      } 

      enemyXScreenNew[i] = enemyXMap[i] - screenXNew;
      enemyYScreenNew[i] = enemyYMap[i] - screenYNew;
      drawEnemies(i);
    }
  }  
}

// Initialize enemy movement (for those enemies moving between two points)
void initializeEnemyMovement()
{
  for(i = 0; i < 16; i++)
  {
    if(enemyStatus[i] > 0)
    {
      // Cloud
      if(enemyType[i] == 1)
      {  
        // Nothing to do
      }
      
      // Fireball flying left
      if(enemyType[i] == 2)
      {  
        // Nothing to do
      }
      
      // Fireball flying right
      if(enemyType[i] == 3)
      {  
        // Nothing to do
      }
      
      // Big Bird
      if(enemyType[i] == 4)
      {
        j = enemyX1[i];
        k = enemyY1[i];
        while(k > 0 && levelMap[(tileNumberX * k) + j] == 0)
        {
          k--;
        }
        if(k == 0)
        {
          enemyY1[i] = k * 8;
        }
        else
        {
          enemyY1[i] = (k + 1) * 8;
        }
        j = enemyX2[i];
        k = enemyY2[i];
        while(k < tileNumberY && levelMap[(tileNumberX * k) + j] == 0)
        {
          k++;
        }
        if(levelMap[(tileNumberX * k) + j] == 1)
        {
          enemyY2[i] = (k * 8) + 5;
        }
        else if(levelMap[(tileNumberX * k) + j] == 2)
        {
          enemyY2[i] = (k * 8) + 1;
        }
        else
        {
          enemyY2[i] = ((k - 1) * 8) + 6;
        }
      }
      
      // Parrot
      if(enemyType[i] == 5)
      {  
        j = enemyX1[i];
        k = enemyY1[i];
        while(j > 0 && (levelMap[(tileNumberX * k) + j] == 0 || levelMap[(tileNumberX * k) + j] == 1 || levelMap[(tileNumberX * k) + j] == 2))
        {
          j--;
        }
        if(levelMap[(tileNumberX * k) + j] == 0 || levelMap[(tileNumberX * k) + j] == 1 || levelMap[(tileNumberX * k) + j] == 2)
        {
          enemyX1[i] = (j * 8);
        }
        else if(levelMap[(tileNumberX * k) + j] == 3 || levelMap[(tileNumberX * k) + j] == 5  || levelMap[(tileNumberX * k) + j] == 7)
        {
          enemyX1[i] = (j * 8) + 1;
        }
        else
        {
          enemyX1[i] = ((j + 1) * 8);
        }
        
        j = enemyX2[i];
        k = enemyY2[i];
        while(j < tileNumberX && (levelMap[(tileNumberX * k) + j] == 0 || levelMap[(tileNumberX * k) + j] == 1 || levelMap[(tileNumberX * k) + j] == 2))
        {
          j++;
        }
        if(levelMap[(tileNumberX * k) + j] == 0 || levelMap[(tileNumberX * k) + j] == 1 || levelMap[(tileNumberX * k) + j] == 2)
        {
          enemyX2[i] = ((j - 1) * 8) + 2;
        }
        else if(levelMap[(tileNumberX * k) + j] == 4 || levelMap[(tileNumberX * k) + j] == 6  || levelMap[(tileNumberX * k) + j] == 8)
        {
          enemyX2[i] = (j * 8) + 1;
        }
        else
        {
          enemyX2[i] = ((j - 1) * 8) + 2;
        }
      }
      
      // Swirl
      if(enemyType[i] == 6)
      {  
        // Nothing to do
      }
      
      // Humming top
      if(enemyType[i] == 7)
      {  
        // Nothing to do
      }
      
      // Devil
      if(enemyType[i] == 8)
      {  
        // Nothing to do
      }
      
      // Hornet
      if(enemyType[i] == 10)
      {  
        j = enemyX1[i];
        k = enemyY1[i];
        while(j > 0 && (levelMap[(tileNumberX * k) + j] == 0 || levelMap[(tileNumberX * k) + j] == 1))
        {
          j--;
        }
        if(j == 0 && (levelMap[(tileNumberX * k) + j] == 0 || levelMap[(tileNumberX * k) + j] == 1))
        {
          enemyX1[i] = j * 8;
        }
        else
        {
          enemyX1[i] = (j + 1) * 8;
        }
        j = enemyX2[i];
        k = enemyY2[i];
        while(j < tileNumberX && (levelMap[(tileNumberX * k) + j] == 0 || levelMap[(tileNumberX * k) + j] == 1))
        {
          j++;
        }
        if(levelMap[(tileNumberX * k) + j] == 4)
        {
          enemyX2[i] = (j * 8) + 3;
        }
        else
        {
          enemyX2[i] = ((j - 1) * 8) + 4;
        }
      }

      // Spider
      if(enemyType[i] == 11)
      {  
        j = enemyX1[i];
        k = enemyY1[i];
        while(j > 0 && levelMap[(tileNumberX * k) + j] == 1)
        {
          j--;
        }
        if(j == 0 && levelMap[(tileNumberX * k) + j] == 1)
        {
          enemyX1[i] = (j * 8);
        }
        else if(levelMap[(tileNumberX * k) + j] == 5)
        {
          enemyX1[i] = (j * 8) + 1;
        }
        else
        {
          enemyX1[i] = ((j + 1) * 8) + 1;
        }
        j = enemyX2[i];
        k = enemyY2[i];
        while(j < tileNumberX && levelMap[(tileNumberX * k) + j] == 1)
        {
          j++;
        }
        if(levelMap[(tileNumberX * k) + j] == 6)
        {
          enemyX2[i] = (j * 8) + 1;          
        }
        else
        {
          enemyX2[i] = ((j - 1) * 8) + 1;
        }
      }

      // Spider upside down
      if(enemyType[i] == 12)
      {  
        j = enemyX1[i];
        k = enemyY1[i];
        while(j > 0 && (levelMap[(tileNumberX * (k - 1)) + j] == 1 || levelMap[(tileNumberX * (k - 1)) + j] == 5))
        {
          j--;
        }
        if(j == 0 && (levelMap[(tileNumberX * (k - 1)) + j] == 1 || levelMap[(tileNumberX * (k - 1)) + j] == 5))
        {
          enemyX1[i] = (j * 8);
        }
        else
        {
          enemyX1[i] = ((j + 1) * 8) + 1;
        }
        j = enemyX2[i];
        k = enemyY2[i];
        while(j < tileNumberX && (levelMap[(tileNumberX * (k - 1)) + j] == 1 || levelMap[(tileNumberX * (k - 1)) + j] == 6))
        {
          j++;
        }
        enemyX2[i] = ((j - 1) * 8) + 1;
      }
      
      // Vertical moving laser
      if(enemyType[i] == 14)
      {  
        j = enemyX1[i];
        k = enemyY1[i];
        while(k > 0 && levelMap[(tileNumberX * k) + j] == 0)
        {
          k--;
        }
        if(levelMap[(tileNumberX * k) + j] == 0)
        {
          enemyY1[i] = k * 8;
        }
        else
        {
          enemyY1[i] = (k + 1) * 8;
        }
        j = enemyX2[i];
        k = enemyY2[i];
        while(k < tileNumberY && levelMap[(tileNumberX * k) + j] == 0)
        {
          k++;
        }
        if(levelMap[(tileNumberX * k) + j] == 1)
        {
          enemyY2[i] = (k * 8) + 6;
        }
        else
        {
          enemyY2[i] = ((k - 1) * 8) + 7;
        }
      }      
      
      // Horizontal moving laser
      if(enemyType[i] == 15)
      {  
        j = enemyX1[i];
        k = enemyY1[i];
        while(j > 0 && levelMap[(tileNumberX * k) + j] == 1)
        {
          j--;
        }
        if(j == 0 && levelMap[(tileNumberX * k) + j] == 1)
        {
          enemyX1[i] = j * 8;
        }
        else if(levelMap[(tileNumberX * k) + j] == 5)
        {
          enemyX1[i] = (j * 8) + 1;
        }
        else
        {
          enemyX1[i] = (j + 1) * 8;
        }
        j = enemyX2[i];
        k = enemyY2[i];
        while(j < tileNumberX && levelMap[(tileNumberX * k) + j] == 1)
        {
          j++;
        }
        if(levelMap[(tileNumberX * k) + j] == 6)
        {
          enemyX2[i] = (j * 8) + 5;
        }
        else
        {
          enemyX2[i] = ((j - 1) * 8) + 6;
        }
      }      

      // Vertical barrier
      if(enemyType[i] == 16)
      {
        j = enemyX1[i];
        k = enemyY1[i];
        while(k > 0 && levelMap[(tileNumberX * k) + j] == 0)
        {
          k--;
        }
        if(k == 0)
        {
          enemyY1[i] = k * 8;
        }
        else
        {
          enemyY1[i] = ((k + 1) * 8) + 1;
        }
        j = enemyX2[i];
        k = enemyY2[i];
        while(k < tileNumberY && levelMap[(tileNumberX * k) + j] == 0)
        {
          k++;
        }
        if(levelMap[(tileNumberX * k) + j] == 1)
        {
          enemyY2[i] = (k * 8) + 3;
        }
        else if(levelMap[(tileNumberX * k) + j] == 2)
        {
          enemyY2[i] = (k * 8) + 2;
        }
        else
        {
          enemyY2[i] = ((k - 1) * 8) + 4;
        }
      }
      
      // Horizontal barrier
      if(enemyType[i] == 17)
      {
        j = enemyX1[i];
        k = enemyY1[i];
        while(j > 0 && (levelMap[(tileNumberX * k) + j] == 0 || levelMap[(tileNumberX * k) + j] == 1))
        {
          j--;
        }
        if(j == 0 && (levelMap[(tileNumberX * k) + j] == 0 || levelMap[(tileNumberX * k) + j] == 1))
        {
          enemyX1[i] = j * 8;
        }
        else if(levelMap[(tileNumberX * k) + j] == 3 || levelMap[(tileNumberX * k) + j] == 5)
        {
          enemyX1[i] = (j * 8) + 2;
        }
        else
        {
          enemyX1[i] = (j + 1) * 8;
        }
        j = enemyX2[i];
        k = enemyY2[i];
        while(j < tileNumberX && (levelMap[(tileNumberX * k) + j] == 0 || levelMap[(tileNumberX * k) + j] == 1))
        {
          j++;
        }
        if(levelMap[(tileNumberX * k) + j] == 4 || levelMap[(tileNumberX * k) + j] == 6)
        {
          enemyX2[i] = (j * 8) + 2;
        }
        else
        {
          enemyX2[i] = ((j - 1) * 8) + 6;
        }
      }
    }
  }
}

// Draw all not collected flowers
void drawFlowers()
{
  for(i = 0; i < 16; i++)
  {
    if(flowerStatus[i] > 0)
    {
      flowerXScreenNew[i] = flowerXMap[i] - screenXNew;
      flowerYScreenNew[i] = flowerYMap[i] - screenYNew;

      if(flowerXScreen[i] > -8 && flowerXScreen[i] < 32 && flowerYScreen[i] > -8 && flowerYScreen[i] < 32)
      {     
        x1 = flowerXScreen[i];
        y1 = flowerYScreen[i];
        x2 = flowerXScreenNew[i];
        y2 = flowerYScreenNew[i];
                
        // Remove flowers at old position
        matrix.drawPixel(x1 + 1, y1, matrix.Color333(0, 0, 0));
        matrix.drawPixel(x1, y1 + 1, matrix.Color333(0, 0, 0));
        matrix.drawPixel(x1 + 1, y1 + 1, matrix.Color333(0, 0, 0));
        matrix.drawPixel(x1 + 2, y1 + 1, matrix.Color333(0, 0, 0));
        matrix.drawPixel(x1 + 1, y1 + 2, matrix.Color333(0, 0, 0));
        matrix.drawPixel(x1 + 1, y1 + 3, matrix.Color333(0, 0, 0));
                
        playfield[x1 + 9][y1 + 8] = 0;
        playfield[x1 + 8][y1 + 9] = 0;
        playfield[x1 + 9][y1 + 9] = 0;
        playfield[x1 + 10][y1 + 9] = 0;
        playfield[x1 + 9][y1 + 10] = 0;
        playfield[x1 + 9][y1 + 11] = 0;
        
        if(flowerStatus[i] == 3)
        {
          flowerStatus[i] = 0;
          flowerCounter--;
        }

        // Draw flowers at new position
        if(flowerStatus[i] == 1 || flowerStatus[i] == 2)
        {
          if(flowerXScreenNew[i] > -8 && flowerXScreenNew[i] < 32 && flowerYScreenNew[i] > -8 && flowerYScreenNew[i] < 32)
          {
            if(flowerStatus[i] == 1)
            {
              matrix.drawPixel(x2 + 1, y2, matrix.Color333(1, 1, 0));
              matrix.drawPixel(x2, y2 + 1, matrix.Color333(1, 1, 0));
              matrix.drawPixel(x2 + 1, y2 + 1, matrix.Color333(0, 0, 1));
              if(animationCounter % 8 < 2) { matrix.drawPixel(x2 + 1, y2 + 1, matrix.Color333(0, 0, 5)); }
              matrix.drawPixel(x2 + 2, y2 + 1, matrix.Color333(1, 1, 0));
              matrix.drawPixel(x2 + 1, y2 + 2, matrix.Color333(1, 1, 0));
              matrix.drawPixel(x2 + 1, y2 + 3, matrix.Color333(0, 1, 0));
            }
            if(flowerStatus[i] == 2)
            {
              // Extralife flower
              matrix.drawPixel(x2 + 1, y2, matrix.Color333(1, 0, 0));
              matrix.drawPixel(x2, y2 + 1, matrix.Color333(1, 0, 0));
              matrix.drawPixel(x2 + 1, y2 + 1, matrix.Color333(0, 1, 1));
              matrix.drawPixel(x2 + 2, y2 + 1, matrix.Color333(1, 0, 0));
              matrix.drawPixel(x2 + 1, y2 + 2, matrix.Color333(1, 0, 0));
              matrix.drawPixel(x2 + 1, y2 + 3, matrix.Color333(0, 1, 0));
              if(animationCounter % 8 < 2) { matrix.drawPixel(x2 + 1, y2, matrix.Color333(5, 0, 0)); }
              else if(animationCounter % 8 < 4) { matrix.drawPixel(x2, y2 + 1, matrix.Color333(5, 0, 0)); }
              else if(animationCounter % 8 < 6) { matrix.drawPixel(x2 + 2, y2 + 1, matrix.Color333(5, 0, 0)); }
              else if(animationCounter % 8 < 8) { matrix.drawPixel(x2 + 1, y2 + 2,  matrix.Color333(5, 0, 0)); }
            }
            playfield[x2 + 9][y2 + 8] = i + 3;
            playfield[x2 + 8][y2 + 9] = i + 3;
            playfield[x2 + 9][y2 + 9] = i + 3;
            playfield[x2 + 10][y2 + 9] = i + 3;
            playfield[x2 + 9][y2 + 10] = i + 3;
            playfield[x2 + 9][y2 + 11] = i + 3;
          }
        }
      }  
      flowerXScreen[i] = flowerXScreenNew[i];
      flowerYScreen[i] = flowerYScreenNew[i];
    }
  }  
}

// Draw player
void drawPlayer()
{
  // Remove player at old position
  matrix.drawPixel(playerXScreen + 1, playerYScreen, matrix.Color333(0, 0, 0));
  matrix.drawPixel(playerXScreen + 2, playerYScreen, matrix.Color333(0, 0, 0));
  matrix.drawPixel(playerXScreen + 3, playerYScreen, matrix.Color333(0, 0, 0));
  matrix.drawPixel(playerXScreen + 4, playerYScreen, matrix.Color333(0, 0, 0));
  matrix.drawPixel(playerXScreen, playerYScreen + 1, matrix.Color333(0, 0, 0));
  matrix.drawPixel(playerXScreen + 1, playerYScreen + 1, matrix.Color333(0, 0, 0));
  matrix.drawPixel(playerXScreen + 2, playerYScreen + 1, matrix.Color333(0, 0, 0));
  matrix.drawPixel(playerXScreen + 3, playerYScreen + 1, matrix.Color333(0, 0, 0));
  matrix.drawPixel(playerXScreen + 4, playerYScreen + 1, matrix.Color333(0, 0, 0));
  matrix.drawPixel(playerXScreen + 5, playerYScreen + 1, matrix.Color333(0, 0, 0));
  matrix.drawPixel(playerXScreen, playerYScreen + 2, matrix.Color333(0, 0, 0));
  matrix.drawPixel(playerXScreen + 1, playerYScreen + 2, matrix.Color333(0, 0, 0));
  matrix.drawPixel(playerXScreen + 2, playerYScreen + 2, matrix.Color333(0, 0, 0));
  matrix.drawPixel(playerXScreen + 3, playerYScreen + 2, matrix.Color333(0, 0, 0));
  matrix.drawPixel(playerXScreen + 4, playerYScreen + 2, matrix.Color333(0, 0, 0));
  matrix.drawPixel(playerXScreen + 5, playerYScreen + 2, matrix.Color333(0, 0, 0));
  matrix.drawPixel(playerXScreen, playerYScreen + 3, matrix.Color333(0, 0, 0));
  matrix.drawPixel(playerXScreen + 1, playerYScreen + 3, matrix.Color333(0, 0, 0));
  matrix.drawPixel(playerXScreen + 2, playerYScreen + 3, matrix.Color333(0, 0, 0));
  matrix.drawPixel(playerXScreen + 3, playerYScreen + 3, matrix.Color333(0, 0, 0));
  matrix.drawPixel(playerXScreen + 4, playerYScreen + 3, matrix.Color333(0, 0, 0));
  matrix.drawPixel(playerXScreen + 5, playerYScreen + 3, matrix.Color333(0, 0, 0));
  
  // Draw player at new position
  if(playerDirectionNew == RIGHT)
  {
    matrix.drawPixel(playerXScreenNew + 3, playerYScreenNew, matrix.Color333(5, 2, 0));
    if(flapAnimation == 0 || flapAnimation == 5)
    {
      matrix.drawPixel(playerXScreenNew + 1, playerYScreenNew + 1, matrix.Color333(0, 0, 5));
      matrix.drawPixel(playerXScreenNew + 2, playerYScreenNew + 1, matrix.Color333(0, 0, 5));
    }
    matrix.drawPixel(playerXScreenNew + 3, playerYScreenNew + 1, matrix.Color333(3, 0, 3));
    matrix.drawPixel(playerXScreenNew + 4, playerYScreenNew + 1, matrix.Color333(5, 0, 5));
    matrix.drawPixel(playerXScreenNew + 5, playerYScreenNew + 1, matrix.Color333(1, 1, 7));
    if(flapAnimation == 2 || flapAnimation ==4)
    {
      matrix.drawPixel(playerXScreenNew + 1, playerYScreenNew + 2, matrix.Color333(0, 0, 5));
      matrix.drawPixel(playerXScreenNew + 2, playerYScreenNew + 2, matrix.Color333(0, 0, 5));
      // Sound: Flap animation
      if(!jingleActive){ tone(audio, 100 + random(100),20); }
    }
    else
    {
      matrix.drawPixel(playerXScreenNew + 1, playerYScreenNew + 2, matrix.Color333(0, 0, 1));
      matrix.drawPixel(playerXScreenNew + 2, playerYScreenNew + 2, matrix.Color333(0, 0, 1));      
    }
    matrix.drawPixel(playerXScreenNew + 3, playerYScreenNew + 2, matrix.Color333(3, 0, 3));
    matrix.drawPixel(playerXScreenNew + 4, playerYScreenNew + 2, matrix.Color333(0, 0, 1));
    if(flapAnimation == 3)
    {
      matrix.drawPixel(playerXScreenNew + 1, playerYScreenNew + 3, matrix.Color333(0, 0, 5));
      matrix.drawPixel(playerXScreenNew + 2, playerYScreenNew + 3, matrix.Color333(0, 0, 5));
      // Sound: Flap animation
      if(!jingleActive){ tone(audio, 50 + random(50), 20); }
    }
    if(landed)
    {
      matrix.drawPixel(playerXScreenNew, playerYScreenNew + 3, matrix.Color333(0, 0, 1));
      matrix.drawPixel(playerXScreenNew + 3, playerYScreenNew + 3, matrix.Color333(0, 3, 0));            
    }
    else
    {
      matrix.drawPixel(playerXScreenNew, playerYScreenNew + 2, matrix.Color333(0, 0, 1));
      matrix.drawPixel(playerXScreenNew + 4, playerYScreenNew + 3, matrix.Color333(0, 3, 0));      
    }
  }
  else
  {
    matrix.drawPixel(playerXScreenNew + 2, playerYScreenNew, matrix.Color333(5, 2, 0));
    matrix.drawPixel(playerXScreenNew, playerYScreenNew + 1, matrix.Color333(1, 1, 7));
    matrix.drawPixel(playerXScreenNew + 1, playerYScreenNew + 1, matrix.Color333(5, 0, 5));
    matrix.drawPixel(playerXScreenNew + 2, playerYScreenNew + 1, matrix.Color333(3, 0, 3));
    if(flapAnimation == 0 || flapAnimation == 5)
    {
      matrix.drawPixel(playerXScreenNew + 3, playerYScreenNew + 1, matrix.Color333(0, 0, 5));
      matrix.drawPixel(playerXScreenNew + 4, playerYScreenNew + 1, matrix.Color333(0, 0, 5));
    }
    matrix.drawPixel(playerXScreenNew + 1, playerYScreenNew + 2, matrix.Color333(0, 0, 1));
    matrix.drawPixel(playerXScreenNew + 2, playerYScreenNew + 2, matrix.Color333(3, 0, 3));
    if(flapAnimation == 2 || flapAnimation == 4)
    {
      matrix.drawPixel(playerXScreenNew + 3, playerYScreenNew + 2, matrix.Color333(0, 0, 5));
      matrix.drawPixel(playerXScreenNew + 4, playerYScreenNew + 2, matrix.Color333(0, 0, 5));
      // Sound: Flap animation
      if(!jingleActive){ tone(audio, 100 + random(100),20); }
    }
    else
    {
      matrix.drawPixel(playerXScreenNew + 3, playerYScreenNew + 2, matrix.Color333(0, 0, 1));
      matrix.drawPixel(playerXScreenNew + 4, playerYScreenNew + 2, matrix.Color333(0, 0, 1));      
    }
    if(flapAnimation == 3)
    {
      matrix.drawPixel(playerXScreenNew + 3, playerYScreenNew + 3, matrix.Color333(0, 0, 5));
      matrix.drawPixel(playerXScreenNew + 4, playerYScreenNew + 3, matrix.Color333(0, 0, 5));      
      // Sound: Flap animation
      if(!jingleActive){ tone(audio, 50 + random(50), 20); }
    }
    if(landed)
    {
      matrix.drawPixel(playerXScreenNew + 5, playerYScreenNew + 3, matrix.Color333(0, 0, 1));
      matrix.drawPixel(playerXScreenNew + 2, playerYScreenNew + 3, matrix.Color333(0, 3, 0));            
    }
    else
    {
      matrix.drawPixel(playerXScreenNew + 5, playerYScreenNew + 2, matrix.Color333(0, 0, 1));
      matrix.drawPixel(playerXScreenNew + 1, playerYScreenNew + 3, matrix.Color333(0, 3, 0));      
    }
  }
}

// Check whether movement to the left is possible
boolean directionLeftFree()
{
  if(playfield[playerXScreenNew + 9][playerYScreenNew + 8] == 1){ return false; }
  if(playfield[playerXScreenNew + 7][playerYScreenNew + 9] == 1){ return false; }
  if(playfield[playerXScreenNew + 8][playerYScreenNew + 10] == 1){ return false; }
  if(playfield[playerXScreenNew + 8][playerYScreenNew + 11] == 1){ return false; }
  return true;
}

// Check whether movement to the right is possible
boolean directionRightFree()
{
  if(playfield[playerXScreenNew + 12][playerYScreenNew + 8] == 1){ return false; }
  if(playfield[playerXScreenNew + 14][playerYScreenNew + 9] == 1){ return false; }
  if(playfield[playerXScreenNew + 13][playerYScreenNew + 10] == 1){ return false; }
  if(playfield[playerXScreenNew + 13][playerYScreenNew + 11] == 1){ return false; }
  return true;
}

// Check whether proceeding of jump is possible
boolean directionUpFree()
{
  if(playerDirection == RIGHT)
  {
    if(playfield[playerXScreenNew + 8][playerYScreenNew + 9] == 1){ return false; }
    if(playfield[playerXScreenNew + 9][playerYScreenNew + 8] == 1){ return false; }
    if(playfield[playerXScreenNew + 10][playerYScreenNew + 8] == 1){ return false; }
    if(playfield[playerXScreenNew + 11][playerYScreenNew + 7] == 1){ return false; }
    if(playfield[playerXScreenNew + 12][playerYScreenNew + 9] == 1){ return false; }
    if(playfield[playerXScreenNew + 13][playerYScreenNew + 8] == 1){ return false; }
  }
  else
  {
    if(playfield[playerXScreenNew + 8][playerYScreenNew + 8] == 1){ return false; }
    if(playfield[playerXScreenNew + 9][playerYScreenNew + 9] == 1){ return false; }
    if(playfield[playerXScreenNew + 10][playerYScreenNew + 7] == 1){ return false; }
    if(playfield[playerXScreenNew + 11][playerYScreenNew + 8] == 1){ return false; }
    if(playfield[playerXScreenNew + 12][playerYScreenNew + 8] == 1){ return false; }
    if(playfield[playerXScreenNew + 13][playerYScreenNew + 9] == 1){ return false; }
  }
  return true;
}

// Check whether proceeding of falling is possible
boolean directionDownFree()
{
  if(playfield[playerXScreenNew + 8][playerYScreenNew + 11] == 1){ return false; }
  if(playfield[playerXScreenNew + 9][playerYScreenNew + 12] == 1){ return false; }
  if(playfield[playerXScreenNew + 10][playerYScreenNew + 12] == 1){ return false; }
  if(playfield[playerXScreenNew + 11][playerYScreenNew + 12] == 1){ return false; }
  if(playfield[playerXScreenNew + 12][playerYScreenNew + 12] == 1){ return false; }
  if(playfield[playerXScreenNew + 13][playerYScreenNew + 11] == 1){ return false; }
  return true;
}

// Moves player (checks also the joystick)
void movePlayer()
{
  // Simulate friction
  if(xSpeed < 0.0)
  { 
    xSpeed = xSpeed + 0.025;
    if(xSpeed > 0.0){ xSpeed = 0.0; }
  }
  else if(xSpeed > 0.0)
  { 
    xSpeed = xSpeed - 0.025;
    if(xSpeed < 0.0){ xSpeed = 0.0; }
  }

  // Simulate gravitiy
  if(ySpeed < 0.5){ ySpeed = ySpeed + 0.035; }

  // Left
  if(joy1Left() && !landed)
  {
    xSpeed = xSpeed - 0.1;
    if(xSpeed < -0.8){ xSpeed = -0.8; }
  }
  // Right
  else if(joy1Right() && !landed)
  {
    xSpeed = xSpeed + 0.1;
    if(xSpeed > 0.8){ xSpeed = 0.8; }    
  }

  // Up
  if((joy1FireA() || joy1FireB()) && flapAnimation == 0 && flapping == false)
  {
    flapping = true;
    flapAnimation = 1;
    ySpeed = ySpeed - 0.5;
    if(ySpeed < -1.0){ ySpeed = -1.0; }
  }

  if((flapAnimation == 0 || flapAnimation > 5) && !(joy1FireA() || joy1FireB()))
  {
    flapping = false;
  }

  // Moving to the left side
  if(xSpeed < 0.0 && int(playerXMap) > 0)
  {
    playerDirectionNew = LEFT;
    xStepCounter = xStepCounter + xSpeed;      
    if(directionLeftFree())
    {
      if(xStepCounter <= -1.0)
      {
        xStepCounter = xStepCounter + 1.0;
        playerXMapNew = playerXMap - 1;
        if(screenX > 0 && playerXScreen < 17)
        {
          screenXNew = screenX - 1;
          redraw = true;
        }
        else
        {
          playerXScreenNew = playerXScreen - 1;
        }
      }
    }
    else
    {
      xSpeed = 0.0;
      xStepCounter = 0.0;
    }
  }
  // Moving to the right side
  else if(xSpeed > 0.0 && int(playerXMap) < (mapWidth - 6))
  {
    playerDirectionNew = RIGHT;
    xStepCounter = xStepCounter + xSpeed;      
    if(directionRightFree())
    {
      if(xStepCounter >= 1.0)
      {
        xStepCounter = xStepCounter - 1.0;
        playerXMapNew = playerXMap + 1;
        if(screenX < (mapWidth - 32) && playerXScreen > 12)
        {
          screenXNew = screenX + 1;
          redraw = true;
        }
        else
        {
          playerXScreenNew = playerXScreen + 1;
        }
      }
    }
    else
    {
      xSpeed = 0.0;
      xStepCounter = 0.0;
    }
  }

  // Moving up
  if(ySpeed < 0.0 && int(playerYMap) > 0)
  {
    yStepCounter = yStepCounter + ySpeed;          
    if(directionUpFree())
    {
      if(yStepCounter <= -1.0)
      {
        yStepCounter = yStepCounter + 1.0;
        playerYMapNew = playerYMap - 1;
        if(screenY > 0 && playerYScreen < 16)
        {
          screenYNew = screenY - 1;
          redraw = true;
        }
        else
        {
          playerYScreenNew = playerYScreen - 1;
        }
      }
    }
    else
    {
      ySpeed = 0.0;
      yStepCounter = 0.0;
    }
  }
  if(ySpeed < 0.0 && int(playerYMap) == 0)
  {
    ySpeed = 0.0;
    yStepCounter = 0.0;
  }
  // Moving down
  if(ySpeed > 0.0 && int(playerYMap) < (mapHeight - 3))
  {
    yStepCounter = yStepCounter + ySpeed;              
    if(directionDownFree())
    {
      if(yStepCounter >= 1.0)
      {
        yStepCounter = yStepCounter - 1.0;
        playerYMapNew = playerYMap + 1;
        if(screenY < (mapHeight - 32) && playerYScreen > 13)
        {
          screenYNew = screenY + 1;
          redraw = true;
        }
        else
        {
          playerYScreenNew = playerYScreen + 1;
        }
      }
    }
    else
    {
      ySpeed = 0.0;
      yStepCounter = 0.0;
    }
  }

  // Flap
  if(flapAnimation > 0)
  {
    flapAnimation++;
    if(flapAnimation == 6){ flapAnimation = 0; }
  }

  if((playerDirectionNew == RIGHT && playfield[playerXScreenNew + 11][playerYScreenNew + 12] == 1) || (playerDirectionNew == LEFT && playfield[playerXScreenNew + 10][playerYScreenNew + 12] == 1))
  {
    landed = true;
  }
  else
  {
    landed = false; 
  }

  // Redraw player if position has changed or animation is incomplete   
  if(redraw || ((playerXScreen != playerXScreenNew) || (playerYScreen != playerYScreenNew) || (playerXMap != playerXMapNew) || (playerYMap != playerYMapNew) || (playerDirection != playerDirectionNew) || (flapAnimation > 0)))
  {
    drawPlayer();  
    playerXScreen = playerXScreenNew;
    playerYScreen = playerYScreenNew;
    playerXMap = playerXMapNew;
    playerYMap = playerYMapNew;
    playerDirection = playerDirectionNew;
  }
}

// Checks whether new player position collides with enemy or flower
byte collisionDetection()
{
  if(playerDirection == RIGHT)
  {
    if(playfield[playerXScreenNew + 11][playerYScreenNew + 8] > 1){ return playfield[playerXScreenNew + 11][playerYScreenNew + 8]; }
    if(playfield[playerXScreenNew + 9][playerYScreenNew + 9] > 1){ return playfield[playerXScreenNew + 9][playerYScreenNew + 9]; }
    if(playfield[playerXScreenNew + 10][playerYScreenNew + 9] > 1){ return playfield[playerXScreenNew + 10][playerYScreenNew + 9]; }
    if(playfield[playerXScreenNew + 12][playerYScreenNew + 9] > 1){ return playfield[playerXScreenNew + 12][playerYScreenNew + 9]; }
    if(playfield[playerXScreenNew + 13][playerYScreenNew + 9] > 1){ return playfield[playerXScreenNew + 13][playerYScreenNew + 9]; }
    if(playfield[playerXScreenNew + 8][playerYScreenNew + 10] > 1){ return playfield[playerXScreenNew + 8][playerYScreenNew + 10]; }
    if(playfield[playerXScreenNew + 9][playerYScreenNew + 10] > 1){ return playfield[playerXScreenNew + 9][playerYScreenNew + 10]; }
    if(playfield[playerXScreenNew + 10][playerYScreenNew + 10] > 1){ return playfield[playerXScreenNew + 10][playerYScreenNew + 10]; }
    if(playfield[playerXScreenNew + 12][playerYScreenNew + 10] > 1){ return playfield[playerXScreenNew + 12][playerYScreenNew + 10]; }
    if(playfield[playerXScreenNew + 11][playerYScreenNew + 11] > 1){ return playfield[playerXScreenNew + 11][playerYScreenNew + 11]; }
  }
  else
  {
    if(playfield[playerXScreenNew + 10][playerYScreenNew + 8] > 1){ return playfield[playerXScreenNew + 10][playerYScreenNew + 8]; }
    if(playfield[playerXScreenNew + 8][playerYScreenNew + 9] > 1){ return playfield[playerXScreenNew + 8][playerYScreenNew + 9]; }
    if(playfield[playerXScreenNew + 9][playerYScreenNew + 9] > 1){ return playfield[playerXScreenNew + 9][playerYScreenNew + 9]; }
    if(playfield[playerXScreenNew + 11][playerYScreenNew + 9] > 1){ return playfield[playerXScreenNew + 11][playerYScreenNew + 9]; }
    if(playfield[playerXScreenNew + 12][playerYScreenNew + 9] > 1){ return playfield[playerXScreenNew + 12][playerYScreenNew + 9]; }
    if(playfield[playerXScreenNew + 9][playerYScreenNew + 10] > 1){ return playfield[playerXScreenNew + 9][playerYScreenNew + 10]; }
    if(playfield[playerXScreenNew + 11][playerYScreenNew + 10] > 1){ return playfield[playerXScreenNew + 11][playerYScreenNew + 10]; }
    if(playfield[playerXScreenNew + 12][playerYScreenNew + 10] > 1){ return playfield[playerXScreenNew + 12][playerYScreenNew + 10]; }
    if(playfield[playerXScreenNew + 13][playerYScreenNew + 10] > 1){ return playfield[playerXScreenNew + 13][playerYScreenNew + 10]; }
    if(playfield[playerXScreenNew + 10][playerYScreenNew + 11] > 1){ return playfield[playerXScreenNew + 10][playerYScreenNew + 11]; }
  }
  return 0;  
}

// Check if collision with enemy is true
boolean checkForLosingLive()
{
  // Collision with enemy
  if(collisionDetection() == 2)
  {
    lives--;
    return true;
  }
  // Player falls out of screen
  if(playerYMap == mapHeight - 3)
  {
    lives--;
    return true;
  }
  return false;  
}

// Check if collision with flower is true (flower collected)
void checkForFlowers()
{
  i = collisionDetection();
  if(i > 2)
  {
    // Normal flower collected
    if(flowerStatus[i - 3] == 1)
    {
      flowers++;
      flowersToFinishLevel--;
      // Sound: Flower collected
      jingleActive = 1;
      notePointer = 0;
      waitingTimeTillNextNote = 0;
    }
    // Extralife flower collected
    else if(flowerStatus[i - 3] == 2)
    {
      if(lives < 9)
      {
        lives++;
      }
      // Sound: Extralife flower collected
      jingleActive = 2;
      notePointer = 0;
      waitingTimeTillNextNote = 0;
    }
    flowerStatus[i - 3] = 3;
  }
}

// Show status screen
void showStatus()
{
  changed = true;
  do
  {
    if(changed)
    {
      // Frame
      matrix.setTextColor(matrix.Color333(2,2,0));
      // Game ends because of no more lives or last level completed (in this case a full screen frame is drawn)
      if(lives < 1 || level == numberOfLevels + 1)
      {
        matrix.fillRect(0, 0, 32, 32, matrix.Color333(0,0,0));
        matrix.drawRect(0, 0, 32, 32, matrix.Color333(1,1,1));
      }
      // Status after loosing live
      else
      {
        matrix.fillRect(2, 6, 28, 19, matrix.Color333(0,0,0));
        matrix.drawRect(2, 6, 28, 19, matrix.Color333(1,0,1));
      }
      // Lives
      matrix.drawPixel(4, 9, matrix.Color333(5, 2, 0));
      matrix.drawPixel(4, 10, matrix.Color333(3, 0, 3));
      matrix.drawPixel(5, 10, matrix.Color333(5, 0, 5));
      matrix.drawPixel(4, 11, matrix.Color333(3, 0, 3));
      matrix.drawPixel(4, 12, matrix.Color333(0, 3, 0));
      matrix.setCursor(7, 8);
      matrix.print(lives);
      // Level
      matrix.drawLine(13, 9, 13, 13, matrix.Color333(3, 3, 3));
      matrix.drawLine(13, 13, 15, 13, matrix.Color333(3, 3, 3));
      matrix.setCursor(17, 8);
      if(level <= numberOfLevels)
      {
        matrix.print(level);
      }
      else
      {
        matrix.print("-");
      }
      // Flowers
      matrix.drawPixel(7, 17, matrix.Color333(1, 1, 0));
      matrix.drawPixel(6, 18, matrix.Color333(1, 1, 0));
      matrix.drawPixel(7, 18, matrix.Color333(0, 0, 3));
      matrix.drawPixel(8, 18, matrix.Color333(1, 1, 0));
      matrix.drawPixel(7, 19, matrix.Color333(1, 1, 0));
      matrix.drawPixel(7, 20, matrix.Color333(0, 1, 0));
      matrix.setCursor(11, 16);
      matrix.print(flowers);
      changed = false;
    }
    
    if(joy1Fire() && joy1Left() && level > 1 && lives > 0)
    {
      level--;
      initializeLevelWithFlowers = true;
      changed = true;
      tone(audio,400,20);
      delay(100);  
    }
    else if(joy1Fire() && joy1Right() && level < numberOfLevels && lives > 0)
    {
      level++;
      initializeLevelWithFlowers = true;
      changed = true;
      tone(audio,400,20);
      delay(200);  
    }
  }
  while(!(joy1Fire() && !joy1Left() && !joy1Right())); 
  tone(audio,400,20);
  delay(200);  
}

// Death animation with flying body parts
void realistischeTodesanimation()
{
  for(int i = 7; i > -1; i--)
  {
    // Draw player at new position
    if(playerDirection == RIGHT)
    {
      matrix.drawPixel(playerXScreen + 3, playerYScreen, matrix.Color333(i, i, i));
      if(flapAnimation == 0 || flapAnimation == 5)
      {
        matrix.drawPixel(playerXScreen + 1, playerYScreen + 1, matrix.Color333(i, i, i));
        matrix.drawPixel(playerXScreen + 2, playerYScreen + 1, matrix.Color333(i, i, i));
      }
      matrix.drawPixel(playerXScreen + 3, playerYScreen + 1, matrix.Color333(i, i, i));
      matrix.drawPixel(playerXScreen + 4, playerYScreen + 1, matrix.Color333(i, i, i));
      matrix.drawPixel(playerXScreen + 5, playerYScreen + 1, matrix.Color333(i, i, i));
      if(flapAnimation == 2 || flapAnimation ==4)
      {
        matrix.drawPixel(playerXScreen + 1, playerYScreen + 2, matrix.Color333(i, i, i));
        matrix.drawPixel(playerXScreen + 2, playerYScreen + 2, matrix.Color333(i, i, i));
      }
      else
      {
        matrix.drawPixel(playerXScreen + 1, playerYScreen + 2, matrix.Color333(i, i, i));
        matrix.drawPixel(playerXScreen + 2, playerYScreen + 2, matrix.Color333(i, i, i));      
      }
      matrix.drawPixel(playerXScreen + 3, playerYScreen + 2, matrix.Color333(i, i, i));
      matrix.drawPixel(playerXScreen + 4, playerYScreen + 2, matrix.Color333(i, i, i));
      if(flapAnimation == 3)
      {
        matrix.drawPixel(playerXScreen + 1, playerYScreen + 3, matrix.Color333(i, i, i));
        matrix.drawPixel(playerXScreen + 2, playerYScreen + 3, matrix.Color333(i, i, i));
      }
      if(landed)
      {
        matrix.drawPixel(playerXScreen, playerYScreen + 3, matrix.Color333(i, i, i));
        matrix.drawPixel(playerXScreen + 3, playerYScreen + 3, matrix.Color333(i, i, i));            
      }
      else
      {
        matrix.drawPixel(playerXScreen, playerYScreen + 2, matrix.Color333(i, i, i));
        matrix.drawPixel(playerXScreen + 4, playerYScreen + 3, matrix.Color333(i, i, i));      
      }
    }
    else
    {
      matrix.drawPixel(playerXScreen + 2, playerYScreen, matrix.Color333(i, i, i));
      matrix.drawPixel(playerXScreen, playerYScreen + 1, matrix.Color333(i, i, i));
      matrix.drawPixel(playerXScreen + 1, playerYScreen + 1, matrix.Color333(i, i, i));
      matrix.drawPixel(playerXScreen + 2, playerYScreen + 1, matrix.Color333(i, i, i));
      if(flapAnimation == 0 || flapAnimation == 5)
      {
        matrix.drawPixel(playerXScreen + 3, playerYScreen + 1, matrix.Color333(i, i, i));
        matrix.drawPixel(playerXScreen + 4, playerYScreen + 1, matrix.Color333(i, i, i));
      }
      matrix.drawPixel(playerXScreen + 1, playerYScreen + 2, matrix.Color333(i, i, i));
      matrix.drawPixel(playerXScreen + 2, playerYScreen + 2, matrix.Color333(i, i, i));
      if(flapAnimation == 2 || flapAnimation ==4)
      {
        matrix.drawPixel(playerXScreen + 3, playerYScreen + 2, matrix.Color333(i, i, i));
        matrix.drawPixel(playerXScreen + 4, playerYScreen + 2, matrix.Color333(i, i, i));
      }
      else
      {
        matrix.drawPixel(playerXScreen + 3, playerYScreen + 2, matrix.Color333(i, i, i));
        matrix.drawPixel(playerXScreen + 4, playerYScreen + 2, matrix.Color333(i, i, i));      
      }
      if(flapAnimation == 3)
      {
        matrix.drawPixel(playerXScreen + 3, playerYScreen + 3, matrix.Color333(i, i, i));
        matrix.drawPixel(playerXScreen + 4, playerYScreen + 3, matrix.Color333(i, i, i));      
      }
      if(landed)
      {
        matrix.drawPixel(playerXScreen + 5, playerYScreen + 3, matrix.Color333(i, i, i));
        matrix.drawPixel(playerXScreen + 2, playerYScreen + 3, matrix.Color333(i, i, i));            
      }
      else
      {
        matrix.drawPixel(playerXScreen + 5, playerYScreen + 2, matrix.Color333(i, i, i));
        matrix.drawPixel(playerXScreen + 1, playerYScreen + 3, matrix.Color333(i, i, i));      
      }
    }
    tone(audio, pow(2,i) * 10, 20);    
    delay(20);
  }
}

// printScrollText: Helper procedure for showScrollText
void printScrollText(int scrollOffset, byte y, byte r, byte g, byte b, boolean draw)
{
  matrix.setCursor(scrollOffset, y);

  if(draw)
  {
    matrix.setTextColor(matrix.Color333(r, g, b));
  }
  else
  { 
    matrix.setTextColor(matrix.Color333(0, 0, 0));
  }

  matrix.println("Part 2 ... Go! Go! Go! Rescue Kenny!");
}

// showScrollText: Part of start sequence
void showScrollText()
{
  byte r, g, b, y;
  r = 1;
  g = 1;
  b = 1;
  y = 4;
  boolean scrollDown = true;

  // Clear screen
  matrix.fillRect(0, 0, 32, 32, matrix.Color333(0, 0, 0));

  int scrollOffset = 32;
  matrix.setTextColor(matrix.Color333(0, 0, 3));

  do
  {
    // Get time of engine loop start point (important for a constant game speed)
    engineLoopStartPoint = millis();

    // Scroll map name
    // Remove old position
    printScrollText(scrollOffset, y, r, g, b, false);

    // Update position
    scrollOffset--;
    if(scrollOffset < -240){ scrollOffset = 32; }
    if(scrollDown)
    { 
      y++;
      if(y == 20){ scrollDown = false; }
    }
    else
    {
      y--;
      if(y == 4){ scrollDown = true; }
    }
 

    // Draw new position 
    if(random(100) < 10)
    { 
      // Color effect
      do
      {
        r = random(4) + 1;
        g = random(4) + 1;
        b = random(4) + 4;
      }
      while((r + g + b) < 3 && (r + g + b) > 5);
    }
    printScrollText(scrollOffset, y, r, g, b, true);

    // SOUND: Show Mode
    tone(audio, (-(scrollOffset - 32) + 90 + random(20)) * 2, 10);

    // Loop until 20 milliseconds has passed since begin of engine loop
    do
    {
    }
    while(abs(millis() - engineLoopStartPoint) < 20);
  }
  while(scrollOffset > -232);  
}

// Start sequence before level 1
void startSequence()
{
  // End screen
  matrix.fillRect(0, 0, 32, 32, matrix.Color333(0,0,0));
  matrix.drawRect(0, 0, 32, 32, matrix.Color333(1,1,1));
  
  // Heart
  matrix.drawCircle(12,6,3,matrix.Color333(3,0,0));  
  matrix.drawCircle(19,6,3,matrix.Color333(3,0,0));  
  matrix.fillRect(9,7,16,4,matrix.Color333(0,0,0));
  matrix.drawLine(9,7,15,13,matrix.Color333(3,0,0));
  matrix.drawLine(22,7,16,13,matrix.Color333(3,0,0));
  
  // Kenny
  matrix.drawPixel(9, 15, matrix.Color333(5, 2, 0));
  matrix.drawPixel(9, 16, matrix.Color333(3, 0, 0));
  matrix.drawPixel(10, 16, matrix.Color333(5, 0, 0));
  matrix.drawPixel(9, 17, matrix.Color333(3, 0, 0));
  matrix.drawPixel(9, 18, matrix.Color333(0, 3, 0));
  
  // Kira
  matrix.drawPixel(22, 15, matrix.Color333(5, 2, 0));
  matrix.drawPixel(21, 16, matrix.Color333(5, 0, 5));
  matrix.drawPixel(22, 16, matrix.Color333(3, 0, 3));
  matrix.drawPixel(22, 17, matrix.Color333(3, 0, 3));
  matrix.drawPixel(22, 18, matrix.Color333(0, 3, 0));
  
  // Rings
  matrix.drawCircle(13,24,4,matrix.Color333(1,1,4));  
  matrix.drawCircle(18,24,4,matrix.Color333(1,1,7));  
  
  delay(2000);

  // Remove frame
  for(byte i = 0; i < 32; i++)
  {
    matrix.drawPixel(i, 0, matrix.Color333(0, 0, 0));  
    tone(audio, 512 - i * 2, 5);
    delay(10);
  }
  for(byte i = 0; i < 32; i++)
  {
    matrix.drawPixel(31, i, matrix.Color333(0, 0, 0));  
    tone(audio, 512 - 64 - i * 2, 5);
    delay(10);
  }
  for(byte i = 0; i < 32; i++)
  {
    matrix.drawPixel(31 - i, 31, matrix.Color333(0, 0, 0));  
    tone(audio, 512 - 128 - i * 2, 5);
    delay(10);
  }
  for(byte i = 0; i < 32; i++)
  {
    matrix.drawPixel(0, 31 - i, matrix.Color333(0, 0, 0));  
    tone(audio, 512 - 192 - i * 2, 5);
    delay(10);
  }
  delay(200);

  // Remove heart
  matrix.drawCircle(12,6,3,matrix.Color333(0,0,0));  
  matrix.drawCircle(19,6,3,matrix.Color333(0,0,0));  
  matrix.fillRect(9,7,16,4,matrix.Color333(0,0,0));
  matrix.drawLine(9,7,15,13,matrix.Color333(0,0,0));
  matrix.drawLine(22,7,16,13,matrix.Color333(0,0,0));
  delay(200);

  // Remove rings
  matrix.drawCircle(13,24,4,matrix.Color333(0,0,0));  
  matrix.drawCircle(18,24,4,matrix.Color333(0,0,0));  
  delay(200);

  // Draw surface
  matrix.drawLine(0,19,131,19,matrix.Color333(0,3,0));
  delay(200);

  // UFO appears
  for(byte i = 0; i < 10; i++)
  {
    // Remove UFO
    matrix.drawLine(9, i - 4, 10, i - 4, matrix.Color333(0,0,0));
    matrix.drawLine(6, i - 3, 13, i - 3, matrix.Color333(0,0,0));
    matrix.drawLine(5, i - 2, 14, i - 2, matrix.Color333(0,0,0));
    matrix.drawLine(7, i - 1, 12, i - 1, matrix.Color333(0,0,0));
    matrix.drawPixel(6, i - 2, matrix.Color333(0,0,0));
    matrix.drawPixel(8, i - 2, matrix.Color333(0,0,0));
    matrix.drawPixel(11, i - 2, matrix.Color333(0,0,0));
    matrix.drawPixel(13, i - 2, matrix.Color333(0,0,0));

    // Draw UFO
    matrix.drawLine(9, i - 3, 10, i - 3, matrix.Color333(0,1,1));
    matrix.drawLine(6, i - 2, 13, i - 2, matrix.Color333(0,2,2));
    matrix.drawLine(5, i - 1, 14, i - 1, matrix.Color333(0,3,3));
    matrix.drawLine(7, i, 12, i, matrix.Color333(0,2,2));
    matrix.drawPixel(6, i - 1, matrix.Color333(3,3,0));
    matrix.drawPixel(8, i - 1, matrix.Color333(5,5,0));
    matrix.drawPixel(11, i - 1, matrix.Color333(5,5,0));
    matrix.drawPixel(13, i - 1, matrix.Color333(3,3,0));

    tone(audio, NOTE_C2, 90);
    delay(100);
    tone(audio, NOTE_D2, 90);
    delay(100);
    tone(audio, NOTE_E2, 190);
    delay(200);
  }

  boolean down = true;
  byte beamY = 10;
  byte kennyY = 15;
  for(byte i = 0; i < 255; i++)
  {
    // Draw beam
    if(random(10) < 7){ matrix.drawPixel(8, beamY,matrix.Color333(random(3) + 1,random(3) + 1,random(3) + 1)); }
    if(random(10) < 7){ matrix.drawPixel(9, beamY,matrix.Color333(random(3) + 1,random(3) + 1,random(3) + 1)); }
    if(random(10) < 7){ matrix.drawPixel(10, beamY,matrix.Color333(random(3) + 1,random(3) + 1,random(3) + 1)); }
    if(random(10) < 7){ matrix.drawPixel(11, beamY,matrix.Color333(random(3) + 1,random(3) + 1,random(3) + 1)); }

    tone(audio, 400 + random(400) + (15 - kennyY) * 40, 3);
    delay(10);

    // Remove beam
    matrix.drawPixel(8, beamY,matrix.Color333(0,0,0));
    matrix.drawPixel(9, beamY,matrix.Color333(0,0,0));
    matrix.drawPixel(10, beamY,matrix.Color333(0,0,0));
    matrix.drawPixel(11, beamY,matrix.Color333(0,0,0));

    // Remove Kenny
    if(kennyY > 9) { matrix.drawPixel(9, kennyY, matrix.Color333(0, 0, 0)); }
    if(kennyY + 1 > 9) { matrix.drawPixel(9, kennyY + 1, matrix.Color333(0, 0, 0)); }
    if(kennyY + 1 > 9) { matrix.drawPixel(10, kennyY + 1, matrix.Color333(0, 0, 0)); }
    if(kennyY + 2 > 9) { matrix.drawPixel(9, kennyY + 2, matrix.Color333(0, 0, 0)); }
    if(kennyY + 3 > 9) { matrix.drawPixel(9, kennyY + 3, matrix.Color333(0, 0, 0)); }

    // Redraw Kenny
    if(kennyY > 9) { matrix.drawPixel(9, kennyY, matrix.Color333(5, 2, 0)); }
    if(kennyY + 1 > 9) { matrix.drawPixel(9, kennyY + 1, matrix.Color333(3, 0, 0)); }
    if(kennyY + 1 > 9) { matrix.drawPixel(10, kennyY + 1, matrix.Color333(5, 0, 0)); }
    if(kennyY + 2 > 9) { matrix.drawPixel(9, kennyY + 2, matrix.Color333(3, 0, 0)); }
    if(kennyY + 3 > 9) { matrix.drawPixel(9, kennyY + 3, matrix.Color333(0, 3, 0)); }

    if(down)
    {
      beamY++;
      if(beamY == 18)
      { 
        down = false;
      }
    }
    else
    {
      beamY--;
      if(beamY == 10)
      { 
        down = true;
        if(i > 64 || i % 8 == 0)
        {
          kennyY--;
        }
      }
    }    
  }
  delay(1000);

  // UFO disappears
  for(i = 0; i < 20; i++)
  {
    matrix.drawLine(9, 6, 10, 6, matrix.Color333(0,1,1));
    matrix.drawLine(6, 7, 13, 7, matrix.Color333(0,2,2));
    matrix.drawLine(5, 8, 14, 8, matrix.Color333(0,3,3));
    matrix.drawLine(7, 9, 12, 9, matrix.Color333(0,2,2));
    matrix.drawPixel(6, 8, matrix.Color333(3,3,0));
    matrix.drawPixel(8, 8, matrix.Color333(5,5,0));
    matrix.drawPixel(11, 8, matrix.Color333(5,5,0));
    matrix.drawPixel(13, 8, matrix.Color333(3,3,0));
    tone(audio, 400 - i * 10, (80 - (i * 4)) / 2);
    delay(80 - (i * 4));
    matrix.drawLine(9, 6, 10, 6, matrix.Color333(0,0,0));
    matrix.drawLine(6, 7, 13, 7, matrix.Color333(0,0,0));
    matrix.drawLine(5, 8, 14, 8, matrix.Color333(0,0,0));
    matrix.drawLine(7, 9, 12, 9, matrix.Color333(0,0,0));
    matrix.drawPixel(6, 8, matrix.Color333(0,0,0));
    matrix.drawPixel(8, 8, matrix.Color333(0,0,0));
    matrix.drawPixel(11, 8, matrix.Color333(0,0,0));
    matrix.drawPixel(13, 8, matrix.Color333(0,0,0));
    tone(audio, 200 - i * 10, (i * 4) / 2);
    delay(i * 4);
  }
  delay(1000);

  showScrollText(); 
  delay(200);  
}

// End sequence after completion of all levels
void endSequence()
{
  matrix.fillRect(0, 0, 32, 32, matrix.Color333(0,0,0));
  matrix.setTextColor(matrix.Color333(3,0,3));
  matrix.setCursor(7, 0);
  matrix.println("2nd");
  matrix.setTextColor(matrix.Color333(3,0,3));
  matrix.setCursor(7, 8);
  matrix.println("Try");
  matrix.setTextColor(matrix.Color333(0,0,3));
  matrix.setCursor(4, 16);
  matrix.println("Well");
  matrix.setTextColor(matrix.Color333(0,0,3));
  matrix.setCursor(4, 24);
  matrix.println("Done");
  for(i = 0; i < 3; i++)
  {
    tone(audio, NOTE_C5, 100);
    delay(100);
    tone(audio, NOTE_E5, 100);
    delay(100);
    tone(audio, NOTE_G5, 100);
    delay(100);        
  }
  tone(audio,1047,400);
  delay(4000);
  
  // End screen
  matrix.fillRect(0, 0, 32, 32, matrix.Color333(0,0,0));
  
  int i = -1;
  do
  {

    // Remove scene from old position
  
    // Kenny
    matrix.drawPixel(9, 15 + i, matrix.Color333(0, 0, 0));
    matrix.drawPixel(9, 16 + i, matrix.Color333(0, 0, 0));
    matrix.drawPixel(10, 16 + i, matrix.Color333(0, 0, 0));
    matrix.drawPixel(9, 17 + i, matrix.Color333(0, 0, 0));
    matrix.drawPixel(9, 18 + i, matrix.Color333(0, 0, 0));
    
    // Kira
    matrix.drawPixel(22, 15 + i, matrix.Color333(0, 0, 0));
    matrix.drawPixel(21, 16 + i, matrix.Color333(0, 0, 0));
    matrix.drawPixel(22, 16 + i, matrix.Color333(0, 0, 0));
    matrix.drawPixel(22, 17 + i, matrix.Color333(0, 0, 0));
    matrix.drawPixel(22, 18 + i, matrix.Color333(0, 0, 0));
    
    // Surface
    matrix.drawLine(0,19 + i,131,19 + i,matrix.Color333(0,0,0));

    // Move scene
    i++;

    // Draw scene to new position
  
    // Kenny
    matrix.drawPixel(9, 15 + i, matrix.Color333(5, 2, 0));
    matrix.drawPixel(9, 16 + i, matrix.Color333(3, 0, 0));
    matrix.drawPixel(10, 16 + i, matrix.Color333(5, 0, 0));
    matrix.drawPixel(9, 17 + i, matrix.Color333(3, 0, 0));
    matrix.drawPixel(9, 18 + i, matrix.Color333(0, 3, 0));
    
    // Kira
    matrix.drawPixel(22, 15 + i, matrix.Color333(5, 2, 0));
    matrix.drawPixel(21, 16 + i, matrix.Color333(5, 0, 5));
    matrix.drawPixel(22, 16 + i, matrix.Color333(3, 0, 3));
    matrix.drawPixel(22, 17 + i, matrix.Color333(3, 0, 3));
    matrix.drawPixel(22, 18 + i, matrix.Color333(0, 3, 0));
    
    // Surface
    matrix.drawLine(0,19 + i,131,19 + i,matrix.Color333(0,3,0));

    if(i == 0)
    {
      // Heart
      matrix.drawCircle(12,6,3,matrix.Color333(3,0,0));  
      matrix.drawCircle(19,6,3,matrix.Color333(3,0,0));  
      matrix.fillRect(9,7,16,4,matrix.Color333(0,0,0));
      matrix.drawLine(9,7,15,13,matrix.Color333(3,0,0));
      matrix.drawLine(22,7,16,13,matrix.Color333(3,0,0));

      // Rings
      matrix.drawCircle(13,25,4,matrix.Color333(1,1,4));  
      matrix.drawCircle(18,25,4,matrix.Color333(1,1,7));  

      delay(4000);
      
      // Remove heart
      matrix.drawCircle(12,6,3,matrix.Color333(0,0,0));  
      matrix.drawCircle(19,6,3,matrix.Color333(0,0,0));  
      matrix.fillRect(9,7,16,4,matrix.Color333(0,0,0));
      matrix.drawLine(9,7,15,13,matrix.Color333(0,0,0));
      matrix.drawLine(22,7,16,13,matrix.Color333(0,0,0));

      // Remove rings
      matrix.drawCircle(13,25,4,matrix.Color333(0,0,0));  
      matrix.drawCircle(18,25,4,matrix.Color333(0,0,0));

      delay(1000);
    }
    else
    {
      delay(150);
    }

  }
  while(i < 17);

  showFirework();

  showStatus();
  initializeLevelWithFlowers = true; // Important for a new start of the game (to have a correct gem setup)
}

// Play jingle if active
void playJingle()
{
  if(waitingTimeTillNextNote > 0)
  {
    waitingTimeTillNextNote--;
  }
  else
  {
    if(jingleActive == 1){ tone(audio, jingle1[notePointer], jingle1[notePointer + 1] * 25); }
    if(jingleActive == 2){ tone(audio, jingle2[notePointer], jingle2[notePointer + 1] * 25); }
    waitingTimeTillNextNote = jingle1[notePointer + 1] * 3;
    notePointer = notePointer + 2;
    if((jingleActive == 1 && notePointer >= jingle1Length) || (jingleActive == 2 && notePointer >= jingle2Length))
    { 
      jingleActive = 0;
      notePointer = 0;
      waitingTimeTillNextNote = 0;
    }
  } 
}

// Firework effects in end sequence

// Init firework
void initFirework()
{
  nextBomb = 0;
  for(int i = 0; i < numberOfBombs; i++)
  {
    bomb[i].bombStatus = 0;
    bomb[i].xPosition = 0.0;
    bomb[i].yPosition = 0.0;
    bomb[i].xSpeed = 0.0;
    bomb[i].ySpeed = 0.0;
    bomb[i].blast = 0;
    bomb[i].fuse = 0;
    bomb[i].numberOfMaroons = 0;    
  }

  nextMaroon = 0;
  for(int i = 0; i < numberOfMaroons; i++)
  {
    maroon[i].maroonStatus = 0;
    maroon[i].xPosition = 0.0;
    maroon[i].yPosition = 0.0;
    maroon[i].xSpeed = 0.0;
    maroon[i].ySpeed = 0.0;
    maroon[i].r = 0;
    maroon[i].g = 0;
    maroon[i].b = 0;
    maroon[i].duration = 0;    
  }
  explosionSound = 0;
}

// Launch a firework bomb
void launchBomb()
{
  // Use a slightly random launch angle and convert it from degree to radian measure
  float angle = (random(31) - 15) * 3.14159 / 180.0;
  float fireSpeed = (100.0 - random(20)) / 100.0;

  if(random(100) < 10 && bomb[nextBomb].bombStatus == 0)
  {
    bomb[nextBomb].bombStatus = 1;
    bomb[nextBomb].xPosition = random(2) + 15.0;
    bomb[nextBomb].yPosition = 32.0;
    bomb[nextBomb].xSpeed = sin(angle) * fireSpeed;
    bomb[nextBomb].ySpeed = cos(angle) * (-fireSpeed);
    bomb[nextBomb].blast = 0;
    bomb[nextBomb].fuse = random(16) + 48;
    bomb[nextBomb].numberOfMaroons = 0;    
    nextBomb++;
    explosionSound = 4;
    if(nextBomb == numberOfBombs){ nextBomb = 0; }
  }
}

// Move firework bombs
void moveBombs()
{
  for(int i = 0; i < numberOfBombs; i++)
  {
    if(bomb[i].bombStatus == 1)
    {
      // Remove bomb from old position
      matrix.drawPixel(bomb[i].xPosition, bomb[i].yPosition, matrix.Color333(0, 0, 0));

      if(bomb[i].fuse == 0)
      {
        segmentation(bomb[i].xPosition, bomb[i].yPosition, bomb[i].xSpeed, bomb[i].ySpeed);
        bomb[i].bombStatus = 0;
      }
      else
      {
        // Move bomb
        bomb[i].xPosition = bomb[i].xPosition + bomb[i].xSpeed;
        bomb[i].yPosition = bomb[i].yPosition + bomb[i].ySpeed;
        bomb[i].ySpeed = bomb[i].ySpeed + 0.02;
        bomb[i].fuse--;

        // Draw bomb at new position
        if(bomb[i].fuse > 4)
        {
          matrix.drawPixel(bomb[i].xPosition, bomb[i].yPosition, matrix.Color333(5, 1, 0));
        }
        else
        {
          matrix.drawPixel(bomb[i].xPosition, bomb[i].yPosition, matrix.Color333(7, 7, 7));          
        }
      }
    }
  }
}

// Let a bomb explode (init maroons)
void segmentation(float x, float y, float xSpeed, float ySpeed)
{
  byte r, g, b;
  r = random(8);
  g = random(8);
  b = random(8);
  float angle;
  float helper = random(360);
  for(byte i = 0; i < 12; i++)
  {
    if(maroon[nextMaroon].maroonStatus == 0)
    {
      maroon[nextMaroon].maroonStatus = 1;
      maroon[nextMaroon].xPosition = x;
      maroon[nextMaroon].yPosition = y;
      angle = random(360) * 3.14159 / 180.0;
      angle = (int(helper + ((360 / 12) * i)) % 360) * 3.14159 / 180.0;
      maroon[nextMaroon].xSpeed = sin(angle) * 0.25;
      maroon[nextMaroon].ySpeed = cos(angle) * (-0.25);
      maroon[nextMaroon].r = r;
      maroon[nextMaroon].g = g;
      maroon[nextMaroon].b = b;
      maroon[nextMaroon].duration = 24;        
      nextMaroon++;
      if(nextMaroon == numberOfMaroons){ nextMaroon = 0; }
    }
  }
  explosionSound = 8;
}

// Move maroons
void moveMaroons()
{
  for(int i = 0; i < numberOfMaroons; i++)
  {
    if(maroon[i].maroonStatus == 1)
    {
      // Remove maroon from old position
      matrix.drawPixel(maroon[i].xPosition, maroon[i].yPosition, matrix.Color333(0, 0, 0));

      if(maroon[i].duration == 0)
      {
        maroon[i].maroonStatus = 0;
      }
      else
      {
        // Move maroon
        maroon[i].xPosition = maroon[i].xPosition + maroon[i].xSpeed;
        maroon[i].yPosition = maroon[i].yPosition + maroon[i].ySpeed;
        maroon[i].ySpeed = maroon[i].ySpeed + 0.02;
        maroon[i].duration--;

        // Draw bomb at new position
        matrix.drawPixel(maroon[i].xPosition, maroon[i].yPosition, matrix.Color333(maroon[i].r, maroon[i].g, maroon[i].b));
      }      
    }
  }  
}

// Show firework
void showFirework()
{
  initFirework();
  
  matrix.fillRect(0, 0, 32, 32, matrix.Color333(0,0,0));
  delay(500);
  
  do
  {
    // Get time of engine loop start point (important for a constant game speed)
    engineLoopStartPoint = millis();
    // Loop until 20 milliseconds has passed since begin of engine loop

    // Launch a new bomb
    launchBomb();

    // Move all active bombs
    moveBombs();

    // Move all active maroons
    moveMaroons();

    // Sound: Mortar
    if(explosionSound > 0 && explosionSound < 4)
    {
      if(explosionSound == 3){ tone(audio, 120 + random(30), 10); }
      else if(explosionSound == 2){ tone(audio, 90 + random(45), 10); }
      else if(explosionSound == 1){ tone(audio, 60 + random(15), 10); }
      explosionSound--;
    }
    // Sound: Bomb explodes
    else if(explosionSound > 3)
    { 
      if(explosionSound == 7){ tone(audio, 240 + random(60), 10); }
      else if(explosionSound == 6){ tone(audio, 120 + random(30), 10);}
      else if(explosionSound == 5){ tone(audio, 60 + random(15), 10);}
      explosionSound--;
      if(explosionSound == 4){ explosionSound = 0; }
    }
    
    do
      {
      }
    while(abs(millis() - engineLoopStartPoint) < 20);

  }
  while(!joy1Fire());  
}

// Main loop
void loop()
{
  mithotronic();
  ledMePlay();

  do
  { 
    title();
    startSequence();
    setupGame();
    do
    {
      showStatus();
      setupLevel();
      reset = false; // Set reset indicator to false
      matrix.fillRect(0, 0, 32, 32, matrix.Color333(0,0,0));
      landed = true;
      drawPlayer();
      do
      {
        // Get time of engine loop start point (important for a constant game speed)
        engineLoopStartPoint = millis();

        // Draw screen
        drawScreen();
        
        // Move player (draws also the player)
        if(animationCounter % 2 == 0)
        {
          movePlayer();
        }
        
        // Draw flowers
        drawFlowers();
        
        // Move enemies
        moveEnemies();
        
        // Check whether flower has been collected
        checkForFlowers();

        // Play jingle
        if(jingleActive > 0)
        { 
          playJingle();
        }
                
        animationCounter++;
    
        // Pause
        if(digitalRead(buttonPause) == LOW)
        {          
          tone(audio,400,20);
          delay(500);
          do
          {
          }
          while((!digitalRead(buttonPause) == LOW) && (!joy1Fire()));
          tone(audio,400,20);
          delay(500);
        }

        // Reset
        if(digitalRead(buttonReset) == LOW)
        {
          reset = true;
          lives = 0;
          break;
        }    

        // Loop until 20 milliseconds has passed since begin of engine loop
        do
          {
          }
        while(abs(millis() - engineLoopStartPoint) < 20);

      }
      while (!checkForLosingLive() && flowersToFinishLevel > 0);

      // Live lost
      if(flowersToFinishLevel > 0)
      {
        realistischeTodesanimation();
        delay(1000);
        initializeLevelWithFlowers = false;
      }
      // Level complete (beam player away)
      else
      {
        j = playerXScreen;
        k = playerYScreen;
        for(i = 0; i < 20; i++)
        {
          playerXScreenNew = j;
          playerYScreenNew = k;
          drawPlayer();
          tone(audio, 400 - i * 10, (80 - (i * 4)) / 2);
          delay(80 - (i * 4));
          playerXScreenNew = -4;
          playerYScreenNew = -4;
          drawPlayer();
          tone(audio, 200 - i * 10, (i * 4) / 2);
          delay(i * 4);
        }
        level++;
        initializeLevelWithFlowers = true;
        matrix.fillRect(0, 0, 32, 32, matrix.Color333(0,0,0));
      }
    }
    while(!reset && level < (numberOfLevels + 1) && lives > 0);

    // All lives lost (Game over sequence)
    if(lives < 1)
    {
      matrix.setTextColor(matrix.Color333(3,0,0));
      matrix.setCursor(4, 8);
      matrix.println("Game");
      matrix.setCursor(4, 16);
      matrix.println("Over");
      tone(audio, NOTE_G4, 200);
      delay(200);
      tone(audio, NOTE_E4, 200);
      delay(200);
      tone(audio, NOTE_C4, 200);
      delay(200);
      tone(audio, NOTE_G3, 400);
      delay(4000);
      showStatus();
    }
    
    // Game completed (End sequence)
    if(level == (numberOfLevels + 1))
    {
      endSequence();
    }
  }
  while(true);
    
} // loop
