package main;

import gui.Piano;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.IOException;
import java.util.ArrayList;
import javax.sound.midi.*;

import utils.DumpReceiver;
import utils.MIDIDispatcher;


public class MIDIPlugsModel implements Constants
{
	private MIDIDispatcher midiDispatcher;
	private static MidiDevice reservedInDevice;
	private static MidiDevice reservedOutDevice;
	private static MidiDevice masterKeyboardDevice;
	
	// Utility field used by the event firing mechanism
	private ArrayList<ActionListener> actionListenerList;
	
	public MidiDevice getReservedOutDevice() { return reservedOutDevice; }
	
	public void setReservedOutDevice(int index) 
	{ 
		// Free the current MIDI out device
		if (reservedOutDevice != null)
		{
			sendAllNotesOff();
			reservedOutDevice.close();
			reservedOutDevice = null;
		}
		// Reserve the new MIDI out device
		reservedOutDevice = midiDispatcher.getOutDevice(index);
		try 
		{
			reservedOutDevice.open();
			System.out.println("Selected MIDI OUT Device: "+reservedOutDevice.getDeviceInfo().getName());
			MIDIDispatcher.setModulatorsMIDIOutPort(reservedOutDevice);
			Piano.setMIDIOutPort(reservedOutDevice);
			// Notify the listener for the change on reserved transmitters
			processEvent(new ActionEvent(this, ActionEvent.ACTION_PERFORMED, "reservedOutDevice"));
			
			// If the user has facilitated MIDI input from a masterkeyboard, link it to the newly assigned
			// MIDI out device
			if (masterKeyboardDevice != null)
			{
				Transmitter transmitter = masterKeyboardDevice.getTransmitter();
				Receiver receiver = reservedOutDevice.getReceiver();
				transmitter.setReceiver(receiver);
			}
		} 
		catch (MidiUnavailableException e) 
		{
		}
	}
	
	public void reserveMasterKeyboardInDevice(int index) 
	{ 
		// Free the current MIDI in device (here: masterkeyboard)
		if (reservedInDevice != null)
		{
			sendAllNotesOff();
			reservedInDevice.close();
			reservedInDevice = null;
		}
		// Reserve the new MIDI in device (i.e., a masterkeyboard)
		reservedInDevice = midiDispatcher.getInDevice(index);
		try 
		{
			reservedInDevice.open();
			System.out.println("Selected MIDI Masterkeyboard Device: "+reservedInDevice.getDeviceInfo().getName());
			masterKeyboardDevice = reservedInDevice;
			if (reservedOutDevice == null) return;
			if (reservedOutDevice.isOpen())
			{
				Transmitter transmitter = reservedInDevice.getTransmitter();
				Receiver receiver = reservedOutDevice.getReceiver();
				transmitter.setReceiver(receiver);
			}
				
			
			// Notify the listener for the change on reserved transmitters
			processEvent(new ActionEvent(this, ActionEvent.ACTION_PERFORMED, "reservedMasterKeyboardInDevice"));
		} 
		catch (MidiUnavailableException e) 
		{
		}
	}
	
	public MidiDevice getReservedInDevice() { return reservedInDevice; }
	
	public void setReservedInDevice(int index) 
	{ 
		reservedInDevice = midiDispatcher.getInDevice(index);
		try 
		{
			reservedInDevice.open();
			System.out.println("Selected MIDI IN Device: "+reservedInDevice.getDeviceInfo().getName());
			// Notify the listener for the change on reserved receivers
			processEvent(new ActionEvent(this, ActionEvent.ACTION_PERFORMED, "reservedInDevice"));
		} 
		catch (MidiUnavailableException e) 
		{
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}
	
	public MIDIDispatcher getMIDIDispatcher() { return this.midiDispatcher; }
	
	public void setMIDIDispatcher(MIDIDispatcher dispatcher)
	{
		midiDispatcher = dispatcher;
	}
	

	public static void sendProgramChange(int program)
	{
		if (reservedOutDevice == null) return;
		try 
		{
			Receiver receiver = reservedOutDevice.getReceiver();
			ShortMessage message = new ShortMessage();
			message.setMessage(ShortMessage.PROGRAM_CHANGE, 0, program, 0);
			receiver.send(message, -1);
		} 
		catch (MidiUnavailableException m) {}
		catch (InvalidMidiDataException i) {};
	}
	
	
	
	public static void getPatchFromSynth(int program)
	{
		if (reservedInDevice == null) return;
		if (reservedOutDevice == null) return;
		
		MidiDeviceTransmitter mdTransmitter = null, transmitter = null;
		MidiDeviceReceiver mdReceiver = null;
		
		// Transmit a program change message for the specified program to the synth
		try 
		{
			mdReceiver = (MidiDeviceReceiver) reservedOutDevice.getReceiver();
		} 
		catch (MidiUnavailableException e) {}
		
		byte byte7 = new Integer( Integer.parseInt("11000000")).byteValue();
		byte byte8 = new Integer( program).byteValue();
		byte[] msg = { byte1, byte2, byte3, byte4, byte5, byte6, byte7, byte8};
		SysexMessage message = new SysexMessage();
        try 
        {
			message.setMessage(msg, 8);
			mdReceiver.send(message, -1);
		} 
        catch (InvalidMidiDataException e1) { System.out.println("Wrong MIDI bytes!");}
        mdReceiver.close();
        
        
        /* NOT YET IMPLEMENTED
        // Get program data from the synth
        DumpReceiver r = new DumpReceiver(System.out);
        try 
		{
			mdTransmitter = (MidiDeviceTransmitter) reservedInDevice.getTransmitter();
		} 
		catch (MidiUnavailableException e3) {}

         mdTransmitter.setReceiver(r);
		try
		{
			System.in.read();
		}
		catch (IOException e) {}
		
		 try {
			Thread.sleep(1000);
		} catch (InterruptedException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		 
		System.out.println("Received "+((DumpReceiver) r).seCount+" sysex messages with a total of "+((DumpReceiver) r).seByteCount+" bytes");
		System.out.println("Received "+((DumpReceiver) r).smCount+" short messages with a total of "+((DumpReceiver) r).smByteCount+" bytes");
		System.out.println("Received a total of "+(((DumpReceiver) r).smByteCount + ((DumpReceiver) r).seByteCount)+" bytes");
		
		//r.close();
		//mdReceiver.close();
		//transmitter.close();
		*/
	}
	
	
	
	public static void sendAllNotesOff()
	{
		if (reservedOutDevice == null) return;
		try 
		{
			Receiver receiver = reservedOutDevice.getReceiver();
			// Send "All Notes Off";
			byte byte7 = new Integer( Integer.parseInt("10110000", 2)).byteValue();	// 1011 0000
			byte byte8 = new Integer( Integer.parseInt("01111011", 2)).byteValue();	// 0111 1011
			byte byte9 = new Integer( Integer.parseInt("00000000", 2)).byteValue(); // 0000 0000
			byte[] msg = { byte1, byte2, byte3, byte4, byte5, byte6, byte7, byte8, byte9};
			SysexMessage message = new SysexMessage();
	        try 
	        {
				message.setMessage(msg, 9);
				receiver.send(message, -1);
			} 
	        catch (InvalidMidiDataException e1) { System.out.println("Wrong MIDI bytes!");}
		} 
		catch (MidiUnavailableException e) {}
	}
	
	// Register an action event listener
	public synchronized void addActionListener(ActionListener l)
	{
		ArrayList<ActionListener> list = actionListenerList == null ? 
				new ArrayList<ActionListener>(2) : (ArrayList) actionListenerList.clone();
		if (!list.contains(l))
		{
			list.add(l);
			actionListenerList = list;
		}
	}
	
	public synchronized void removeActionListener(ActionListener l)
	{
		if (actionListenerList != null && actionListenerList.contains(l))
		{
			actionListenerList.remove(l);
		}
	}
	
	// Fire TickEvent
	private void processEvent(ActionEvent e)
	{
		if (actionListenerList == null) return;
		
		ArrayList list;
		synchronized (this)
		{
			list = (ArrayList) actionListenerList.clone();
		}
		
		for (int i = 0; i < list.size(); i++)
		{
			ActionListener listener = (ActionListener) list.get(i);
			listener.actionPerformed(e);
		}
	}
	
	public MidiDevice getMasterKeyboardDevice() { return masterKeyboardDevice; }
}
