package main;

/*
ILoveMyJuno2 - MIDI SysEx editor for the aJuno line of synthesizers
Copyright (C) 2014 Thomas Laubach

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/



// Lowest possible resolution is (1280, 800)
// If to be steered from a distance: larger JSlider handle, increase the sensitive zone, larger pointer (or a selection box instead of a pointer)
// TODO:	Export sound data in txt
// 		 	Interactive Block Diagram Viewport with comments and descriptions
// 			MIDI Librarian
//			Right-click-Menü: delete, move/copy left/right, compare (zwei Sounds ausgewählt, Anzeige der parameter in Listenform)
//			Right-click-Menü: receiveSoundFromSynth (eine Bank wird nach Bulk Load direkt als neue Bank angefügt), edit, save bank
//			Right-click-Menü: send program to synth, send bank to synth
//			Editor: save current sound
// 			DONE: Export and Import sound banks as SysEx
// 			Simple Arpeggiator
// 			Scene snapshots from entire synth configuration, including the positions of the modulators
// 			Bigger mouse cursor, highlight components under the mouse position
// 			Receive single sounds from synth and adjust the currently edited sound accordingly
// 			Right click on modulator displays inputs and outputs for it including the control change mapping, with small animations
// 			Modify envelope by dragging the spots in the envelope view
// 			Prettify the MIDI dialogs
// 			MIDI Bulk Dump
//			Illustrate by color when a key is played on the keyboard (via mouse or remote!) or a chordal is held
//			Pretty symbols on modifiers like those in the manual
//			Visualize MIDI In from master keyboard and MIDI Out to Juno
//			Placement of modulators requires too high a degree of finesse
//			Small graphical errors during the refresh of the JComponents (broken numbers, overlapping components etc.)
// 			Allow to set the MIDI transmission channel
//			x-y controller
//			x-y-z controller (use jlibfreespace, Loop Pointer, or the Leap Motion Controller, or the Myo armband, or a custom built controller)


import gui.BlockDiagramViewport;
import gui.EnvelopeWidget;
import gui.LibrarianWidget;
import gui.MIDIMasterKeyboardViewController;
import gui.MIDIModulator;
import gui.MIDIPlugsViewController;
import gui.ModulatorsState;
import gui.MovingPanel;
import gui.Piano;
import gui.SignalsViewport;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;

import javax.swing.*;
import javax.swing.UIManager.LookAndFeelInfo;

import javax.swing.border.EmptyBorder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import utils.MIDIDispatcher;
import utils.Presets;

public class ILoveMyJuno2 implements ActionListener, ComponentListener, Constants
{
	private static MIDIPlugsModel mpModel = new MIDIPlugsModel();
	private MIDIPlugsViewController mpController = new MIDIPlugsViewController();
	private MIDIMasterKeyboardViewController mkController = new MIDIMasterKeyboardViewController();
	private MIDIDispatcher dispatcher = new MIDIDispatcher();
	private JMenuBar menuBar = new JMenuBar();
	private JMenuItem reserveMIDIPorts, assignMasterKeyboard, quit, showBlockDiagram, jumpToLibrarian, jumpToEnvelope, openToneBank;
	private MovingPanel controllersPanel = new MovingPanel(1300, 1300);
	public JLabel dragLabel1;
	private JScrollPane pianoScrollPane, controllersScrollPane;
	private JPanel pianoPanel;
	private JFrame frame;
	private JCheckBox holdKey, transposeDown, transposeUp, dontTranspose;
	private JButton allNotesOff, bulkDump, bulkReceive, bulkOpen;
	private JList programSelector;
	private Piano piano;
	private SignalsViewport signalsViewport;
	private BlockDiagramViewport blockDiagramViewport;
	private MIDIModulator dcoNoiseLevel;
	private LibrarianWidget libWidget;
	private MIDIModulator benderRange;
	private MIDIModulator vcfCutoffFreq;
	private MIDIModulator vcfResonance;
	private MIDIModulator vcfLFODepth;
	private MIDIModulator vcfEnvelopeDepth;
	private MIDIModulator vcfEnvelopeMode;
	private MIDIModulator vcfAftertouchSens;
	private MIDIModulator vcfKeyFollow;
	private MIDIModulator lfoRate;
	private MIDIModulator lfoDelay;
	private MIDIModulator dcoRange;
	private MIDIModulator dcoLFODepth;
	private MIDIModulator dcoENVDepth;
	private MIDIModulator dcoENVMode;
	private MIDIModulator dcoAftertouchSens;
	private MIDIModulator dcoPulseWaveform;
	private MIDIModulator dcoSawtoothWaveform;
	private MIDIModulator dcoSubOSCWaveform;
	private MIDIModulator dcoSubOSCLevel;
	private MIDIModulator dcoPWMDepth;
	private MIDIModulator vcaLevel;
	private MIDIModulator vcaENVMode;
	private MIDIModulator vcaAftertouchSens;
	private MIDIModulator chorus;
	private MIDIModulator chorusRate;
	private MIDIModulator hpfCutoffFreq;
	private MIDIModulator envT1;
	private MIDIModulator envT2;
	private MIDIModulator envT3;
	private MIDIModulator envT4;
	private MIDIModulator envL1;
	private MIDIModulator envL2;
	private MIDIModulator envL3;
	private MIDIModulator envKeyFollow;
	private Dimension screenSize = null;
	private Insets frameInsets = null;
	private Librarian librarian;
	
	
	public ILoveMyJuno2()
	{
		RepaintManager.currentManager(null).setDoubleBufferingEnabled(true);
		
		// Some alterations to the Nimbus Look&Feel
		UIManager.put("CheckBoxMenuItem.font", new Font("Arial", Font.PLAIN, 22));
		UIManager.put("MenuItem.font", new Font("Arial", Font.PLAIN, 22));
		UIManager.put("Menu.font", new Font("Arial", Font.PLAIN, 22));
		UIManager.put("Slider.font", new Font("Arial", Font.PLAIN, 22));
		UIManager.put("Button.font", new Font("Arial", Font.PLAIN, 22));
		UIManager.put("nimbusLightBackground",new Color(150, 180, 240));
		UIManager.put("control",new Color(190, 200, 250));
		UIManager.put("nimbusBlueGrey", new Color(120, 140, 180));
		UIManager.put("nimbusBase", new Color(20, 70, 255)); // Incidentially changes the ScrollBar thumb color as well!
		
		// Screen size
		Toolkit toolkit = Toolkit.getDefaultToolkit();
		screenSize = toolkit.getScreenSize();
		
		// Frame
		frame = new JFrame("I Love My Roland alphaJuno");
		frameInsets = frame.getInsets();
		frame.setPreferredSize(new Dimension(screenSize.width, screenSize.height-50));
		frame.setLocation(0, 0);
		frame.setBackground(new Color(200, 200, 210));
		frame.setLayout(new BorderLayout());
		frame.setJMenuBar(menuBar);
		frame.addComponentListener(this);
		frame.setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
				
		// MIDI Devices Dialog
		mpController.setMIDIDispatcher(dispatcher);
		mpModel.setMIDIDispatcher(dispatcher);
		mpController.setModel(mpModel);
		mpController.setVisible(false);
		
		// MIDI MasterKeyboard dialog
		mkController.setMIDIDispatcher(dispatcher);
		mkController.setModel(mpModel);
		mkController.setVisible(false);
		
		// ModulatorsState
		new ModulatorsState();
		
		// Presets
		new Presets();
		
		// Menu
		menuBar.setVisible(false);
		
		JMenu systemMenu = new JMenu("Setup");
		systemMenu.add(reserveMIDIPorts = new JMenuItem("Reserve MIDI Ports"));
		reserveMIDIPorts.addActionListener(this);
		systemMenu.add(assignMasterKeyboard = new JMenuItem("Use Remote Keyboard"));
		assignMasterKeyboard.addActionListener(this);
		
		JMenu fileMenu = new JMenu("File");
		//fileMenu.add(openToneBank = new JMenuItem("Open Tone Bank"));
		//openToneBank.addActionListener(this);
		fileMenu.add(quit = new JMenuItem("Quit"));
		quit.addActionListener(this);
		
		JMenu viewMenu = new JMenu("View");
		viewMenu.add(showBlockDiagram = new JMenuItem("Display Block Diagram"));
		showBlockDiagram.addActionListener(this);
		viewMenu.add(jumpToLibrarian = new JMenuItem("Jump to Librarian"));
		jumpToLibrarian.addActionListener(this);
		viewMenu.add(jumpToEnvelope = new JMenuItem("Jump to Envelope"));
		jumpToEnvelope.addActionListener(this);
		menuBar.add(fileMenu);
		menuBar.add(systemMenu);
		menuBar.add(viewMenu);
		menuBar.doLayout();
		menuBar.setVisible(true);
		
		// Librarian
		librarian = new Librarian();
		
		// UPPER PANEL
		// Controllers panel
		controllersPanel.setPreferredSize(new Dimension(screenSize.width+2500, screenSize.height+485));
		controllersPanel.setMinimumSize(new Dimension(screenSize.width+2500, screenSize.height+485));
		controllersPanel.setLayout(null);
		controllersPanel.setBackground(BG_COL);
		controllersScrollPane = new JScrollPane();
		signalsViewport = new SignalsViewport(controllersPanel);
		signalsViewport.setView(controllersPanel);

		controllersScrollPane.setViewport(signalsViewport);
		controllersScrollPane.setPreferredSize(new Dimension(screenSize.width-frameInsets.left-frameInsets.right, screenSize.height-300));
		controllersScrollPane.getVerticalScrollBar().setCursor(new Cursor(Cursor.HAND_CURSOR));
		controllersScrollPane.getHorizontalScrollBar().setCursor(new Cursor(Cursor.HAND_CURSOR));
		controllersScrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
		controllersScrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
		controllersScrollPane.getVerticalScrollBar().setUnitIncrement(10);
		controllersScrollPane.getHorizontalScrollBar().setUnitIncrement(10);
		controllersScrollPane.doLayout();
		
		
		//UIDefaults scrollBarDefaults = new UIDefaults();
        //controllersScrollPane.getVerticalScrollBar().putClientProperty("Nimbus.Overrides",scrollBarDefaults);
        //controllersScrollPane.getVerticalScrollBar().putClientProperty("Nimbus.Overrides.InheritDefaults",false);
        //controllersScrollPane.getVerticalScrollBar().setUI(new MyScrollBarUI());
        /*
        scrollBarDefaults.put("ScrollBar.minimumThumbSize", new Dimension(35, 35));
        scrollBarDefaults.put("ScrollBar:ScrollBarThumb[Enabled].backgroundPainter", new Painter() 
        {
            public void paint(Graphics2D g, Object c, int w, int h) 
            {
                g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
                g.setStroke(new BasicStroke(2f));
                g.setColor(Color.BLUE);
                g.fillOval(1, 1, w-3, h-3);
                g.setColor(Color.WHITE);
                g.drawOval(1, 1, w-3, h-3);
			}
        });
		*/
		
		
		// LOWER PANEL
		// Transpose-down button
		transposeDown = new JCheckBox("-2 Octaves");
		transposeDown.setForeground(new Color(200, 200, 220));
		transposeDown.setBackground(new Color(10, 10, 15));
		FontMetrics fontMetrics = transposeDown.getFontMetrics(new Font("Arial", Font.PLAIN, 22));
		transposeDown.setMinimumSize(new Dimension((int) fontMetrics.stringWidth("Transpose Down")-20, 40));
		transposeDown.setPreferredSize(new Dimension((int) fontMetrics.stringWidth("Transpose Down")-20, 40));
		transposeDown.setFont(new Font("Arial", Font.PLAIN, 22));
		transposeDown.setCursor(new Cursor(Cursor.HAND_CURSOR));
		transposeDown.addActionListener(this);		
		// Transpose-up button
		transposeUp = new JCheckBox("+1 Octave");
		transposeUp.setForeground(new Color(200, 200, 220));
		transposeUp.setBackground(new Color(10, 10, 15));
		transposeUp.setMinimumSize(transposeDown.getPreferredSize());
		transposeUp.setPreferredSize(transposeDown.getPreferredSize());
		transposeUp.setFont(new Font("Arial", Font.PLAIN, 22));
		transposeUp.setCursor(new Cursor(Cursor.HAND_CURSOR));
		transposeUp.addActionListener(this);	
		// Dummy toggle button for state "do not transpose at all"
		// This button is not shown at all
		dontTranspose = new JCheckBox("not at all"); 
		dontTranspose.setSelected(true);
		
		// Transposition panel
		JPanel transposePanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		transposePanel.setBorder(new EmptyBorder(8, 0, 0, 0));
		transposePanel.setBackground(new Color(10, 10, 15));
		JLabel transposeLabel = new JLabel("Transpose", SwingConstants.LEADING);
		
		transposeLabel.setFont(new Font("Arial", Font.BOLD, 22));
		transposeLabel.setForeground(new Color(190, 190, 220));
		transposeLabel.setBackground(new Color(20, 20, 35));
		transposePanel.add(transposeLabel, BorderLayout.NORTH);
		transposePanel.add(transposeUp, BorderLayout.CENTER);	
		transposePanel.add(transposeDown, BorderLayout.SOUTH);
		transposePanel.setMinimumSize(new Dimension(160, 161)); 
		transposePanel.setPreferredSize(new Dimension(160, 161)); 
		
		// All notes off button
		allNotesOff = new JButton("Notes Off");
		allNotesOff.setFont(new Font("Arial", Font.PLAIN, 22));
		fontMetrics = allNotesOff.getFontMetrics(new Font("Arial", Font.PLAIN, 22));
		allNotesOff.setForeground(new Color(190, 190, 220));
		allNotesOff.setBackground(new Color(20, 20, 35));
		allNotesOff.setMinimumSize(new Dimension((int) fontMetrics.stringWidth("Notes Off")+30, 40));
		allNotesOff.setPreferredSize(new Dimension((int) fontMetrics.stringWidth("Notes Off")+30, 40));
		allNotesOff.addActionListener(this);
		allNotesOff.setSelected(false);
		allNotesOff.setCursor(new Cursor(Cursor.HAND_CURSOR));
		
		
		// Bulk Receive button 
		bulkReceive = new JButton("Receive Tone Bank");
		bulkReceive.setFont(new Font("Arial", Font.PLAIN, 22));
		fontMetrics = allNotesOff.getFontMetrics(new Font("Arial", Font.PLAIN, 22));
		bulkReceive.setForeground(new Color(190, 190, 220));
		bulkReceive.setBackground(new Color(20, 20, 35));
		bulkReceive.setMinimumSize(new Dimension((int) fontMetrics.stringWidth("Transmit Tone Bank")+30, 40));
		bulkReceive.setPreferredSize(new Dimension((int) fontMetrics.stringWidth("Transmit Tone Bank")+30, 40));
		bulkReceive.addActionListener(this);
		bulkReceive.setSelected(false);
		bulkReceive.setCursor(new Cursor(Cursor.HAND_CURSOR));
		bulkReceive.setBorder(null);
		
		// Bulk Send button 
		bulkDump = new JButton("Transmit Tone Bank");
		bulkDump.setFont(new Font("Arial", Font.PLAIN, 22));
		fontMetrics = bulkReceive.getFontMetrics(new Font("Arial", Font.PLAIN, 22));
		bulkDump.setForeground(new Color(190, 190, 220));
		bulkDump.setBackground(new Color(20, 20, 35));
		bulkDump.setMinimumSize(new Dimension((int) fontMetrics.stringWidth("Transmit Tone Bank")+30, 40));
		bulkDump.setPreferredSize(new Dimension((int) fontMetrics.stringWidth("Transmit Tone Bank")+30, 40));
		bulkDump.addActionListener(this);
		bulkDump.setSelected(false);
		bulkDump.setCursor(new Cursor(Cursor.HAND_CURSOR));
		
		// Bulk open button 
		bulkOpen = new JButton("Open Tone Bank");
		bulkOpen.setFont(new Font("Arial", Font.PLAIN, 22));
		fontMetrics = bulkReceive.getFontMetrics(new Font("Arial", Font.PLAIN, 22));
		bulkOpen.setForeground(new Color(190, 190, 220));
		bulkOpen.setBackground(new Color(20, 20, 35));
		bulkOpen.setMinimumSize(new Dimension((int) fontMetrics.stringWidth("Transmit Tone Bank")+30, 40));
		bulkOpen.setPreferredSize(new Dimension((int) fontMetrics.stringWidth("Transmit Tone Bank")+30, 40));
		bulkOpen.addActionListener(this);
		bulkOpen.setSelected(false);
		bulkOpen.setCursor(new Cursor(Cursor.HAND_CURSOR));
		
		// Hold Key checkbox
		holdKey = new JCheckBox("Hold Key", false);
		holdKey.setForeground(new Color(200, 200, 220));
		holdKey.setMinimumSize(allNotesOff.getPreferredSize());
		holdKey.setPreferredSize(allNotesOff.getPreferredSize());
		holdKey.setFont(new Font("Arial", Font.PLAIN, 22));
		holdKey.putClientProperty("JCheckBox.sizeVariant", "large");
		holdKey.addActionListener(this);		
		holdKey.setCursor(new Cursor(Cursor.HAND_CURSOR));
		
		
		// Piano panel
		FlowLayout flowLayout = new FlowLayout(FlowLayout.LEADING, 10, 0);
		pianoPanel = new JPanel(flowLayout);
		pianoPanel.setPreferredSize(new Dimension(screenSize.width+2000, 162 ));
		pianoPanel.setMinimumSize(new Dimension(screenSize.width+2000, 162));
		pianoPanel.setBackground(BG_COL);
		pianoPanel.setBorder(new EmptyBorder(2, 2, 2, 2));
		pianoPanel.setDoubleBuffered(true);
		
		
		// Patch selector
		String[] programListP = Presets.getAllNamesIndexed_BankP();
		String[] programListM = Presets.getAllNamesIndexed_BankM();
		ArrayList<String> programList = new ArrayList<String>(Arrays.asList(programListM));
		programList.addAll(Arrays.asList(programListP));
		String[] list = (String []) programList.toArray(new String[programList.size()]);
		programSelector = new JList<Object>(list);
		programSelector.setForeground(new Color(190, 190, 225));
		programSelector.setFont(new Font("Arial", Font.PLAIN, 22));
		programSelector.setCursor(new Cursor(Cursor.HAND_CURSOR));
		programSelector.addListSelectionListener(new ListSelectionListener(){
				public void valueChanged(ListSelectionEvent e) 
				{
					if (e.getSource() == programSelector)
					{
						int patchNumber = programSelector.getSelectedIndices()[0];
						MIDIPlugsModel.sendProgramChange(patchNumber);
						//MIDIPlugsModel.getPatchFromSynth(patchNumber); // Send a program change via SysEx, then get incoming tone data
					}
				}});
		programSelector.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		programSelector.setFixedCellWidth(250);
		programSelector.setBackground(BG_COL);
		JScrollPane programScrollPane = new JScrollPane(programSelector);
		programScrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
		programScrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
		programScrollPane.setPreferredSize(new Dimension(270, 154));
		programScrollPane.getVerticalScrollBar().setCursor(new Cursor(Cursor.HAND_CURSOR));
		programScrollPane.revalidate();
		
		// Button panel
		JPanel buttonPanel = new JPanel(new FlowLayout());
		buttonPanel.add(holdKey);
		buttonPanel.add(allNotesOff);
		
		buttonPanel.add(transposePanel);
		buttonPanel.setMinimumSize(new Dimension(140, 156)); 
		buttonPanel.setPreferredSize(new Dimension(140, 156)); 
		buttonPanel.setBackground(new Color(10, 10, 15));
		
		// Program selector panel
		JPanel selectorPanel = new JPanel(new BorderLayout());
		selectorPanel.add(programScrollPane, BorderLayout.CENTER);
		
		/*
		// File panel
		JPanel transferPanel = new JPanel(new FlowLayout());
		transferPanel.setPreferredSize(new Dimension(250, 156));
		transferPanel.setBackground(BG_COL);
		transferPanel.setForeground(Color.orange);
		transferPanel.setMinimumSize(new Dimension(250, 156));
		transferPanel.add(bulkReceive);
		transferPanel.add(bulkDump);
		transferPanel.add(bulkOpen);
		*/
		
		// Arpeggiator panel
		JPanel arpPanel = new JPanel();
		arpPanel.setPreferredSize(new Dimension(1500, 156));
		arpPanel.setBackground(BG_COL);
		arpPanel.setForeground(Color.orange);
		arpPanel.setMinimumSize(new Dimension(1500, 156));
		arpPanel.add(new JLabel("BAUSTELLE."));
		
		piano = new Piano();
		pianoPanel.add(piano);
		pianoPanel.add(buttonPanel);
		pianoPanel.add(transposePanel);
		pianoPanel.add(selectorPanel);
		//pianoPanel.add(transferPanel);
		pianoPanel.add(arpPanel);
		
		// ScrollPane for Piano Panel
		pianoScrollPane = new JScrollPane(pianoPanel);
		pianoScrollPane.setPreferredSize(new Dimension(screenSize.width-frameInsets.left-frameInsets.right, 184));
		pianoScrollPane.setMinimumSize(new Dimension(screenSize.width-frameInsets.left-frameInsets.right, 184));
		pianoScrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
		pianoScrollPane.getHorizontalScrollBar().setCursor(new Cursor(Cursor.HAND_CURSOR));
		pianoScrollPane.getVerticalScrollBar().setUnitIncrement(10);
		pianoScrollPane.getHorizontalScrollBar().setUnitIncrement(10);
    
		// Compose the frame
		frame.add(controllersScrollPane, BorderLayout.NORTH);
		frame.add(pianoScrollPane, BorderLayout.SOUTH);
		SwingUtilities.updateComponentTreeUI(menuBar);
		frame.pack();
		
		
		// Force the mouse onto a location where it is not over a JComponent
		// Otherwise a strange effect with the screen repaint is encountered
		try 
		{
			(new Robot()).mouseMove(0, 0);
		} 
		catch (AWTException e1) {};
		frame.setVisible(true);
		populateControllersPanel();
	}
	
	private void populateControllersPanel()
	{   
		// ENVELOPE VIEW
		EnvelopeWidget envWidget = new EnvelopeWidget(controllersPanel, signalsViewport, new Point(2800, 610), envWidgetDefX, envWidgetDefY);
		
		libWidget = new LibrarianWidget(controllersPanel, new Point(300, 760), libWidgetDefX, libWidgetDefY);
		
		Hashtable<Integer, JLabel> labels = null;
		labels = new Hashtable<Integer, JLabel>();
		labels.put(0, createLabel("0", null));
		labels.put(5, createLabel("5", null));
		labels.put(11, createLabel("11", null));
		benderRange = new MIDIModulator(controllersPanel, envWidget, signalsViewport, "BEND RANGE", STD_SLIDER, ENTRY, 12, JSlider.HORIZONTAL, 5, 35, labels);
		
		// VCF BLOCK
		vcfCutoffFreq = new MIDIModulator(controllersPanel, envWidget, signalsViewport, "VCF CUTOFF", STD_SLIDER, VCF, 128, JSlider.HORIZONTAL, 63, 16, null );
		vcfResonance = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "VCF RESO", STD_SLIDER, VCF, 128, JSlider.HORIZONTAL, 63, 17, null);
		vcfLFODepth = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "VCF LFO DEPTH", STD_SLIDER, VCF, 128, JSlider.HORIZONTAL, 63, 18, null);
		vcfEnvelopeDepth = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "VCF ENV DEPTH", STD_SLIDER, VCF, 128, JSlider.HORIZONTAL, 63, 19, null);
		labels = new Hashtable<Integer, JLabel>();
		labels.put(0, createLabel("Normal", null));
		labels.put(1, createLabel("Invert", null));
		labels.put(2, createLabel("Normal w/ Dyn", null));
		labels.put(3, createLabel("Dyn", null));
		vcfEnvelopeMode = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "VCF ENV MODE", STD_SLIDER, VCF,  4, JSlider.VERTICAL, 3, 1, labels);
		labels = new Hashtable<Integer, JLabel>();
		labels.put(0, createLabel("0", null));
		labels.put(5, createLabel("5", null));
		labels.put(10, createLabel("10", null));
		labels.put(15, createLabel("15", null));
		vcfAftertouchSens = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "VCF ATOUCH SENS", STD_SLIDER, VCF, 16, JSlider.HORIZONTAL, 15, 15, labels);
		labels = new Hashtable<Integer, JLabel>();
		labels.put(0, createLabel("0", null));
		labels.put(5, createLabel("5", null));
		labels.put(10, createLabel("10", null));
		labels.put(15, createLabel("15", null));
		vcfKeyFollow = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "VCF KEY FOLLOW", STD_SLIDER, VCF, 16, JSlider.HORIZONTAL, 15, 20, labels);
		labels = null;
		
		lfoRate = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "LFO RATE", STD_SLIDER, LFO, 128, JSlider.HORIZONTAL, 63, 24, null);
		lfoDelay = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "LFO DELAY", STD_SLIDER, LFO, 128, JSlider.HORIZONTAL, 63, 25, null);
			
		// DCO BLOCK
		labels = new Hashtable<Integer, JLabel>();
		labels.put(3, createLabel("32'", null));
		labels.put(2, createLabel("16'", null));
		labels.put(1, createLabel("8'", null));
		labels.put(0, createLabel("4'", null));
		dcoRange = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "DCO RANGE", STD_SLIDER, DCO, 4, JSlider.HORIZONTAL, 3, 6, labels);
		dcoLFODepth = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "DCO LFO DEPTH", STD_SLIDER, DCO,  128, JSlider.HORIZONTAL, 63, 11, null);
		dcoENVDepth = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "DCO ENV DEPTH", STD_SLIDER, DCO, 128, JSlider.HORIZONTAL, 63, 12, null);
		labels = new Hashtable<Integer, JLabel>();
		labels.put(0, createLabel("Normal", null));
		labels.put(1, createLabel("Invert", null));
		labels.put(2, createLabel("Normal w/ Dyn", null));
		labels.put(3, createLabel("Invert w/ Dyn", null));
		dcoENVMode = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "DCO ENV MODE", STD_SLIDER, DCO, 4, JSlider.VERTICAL, 3, 0, labels);
		
		labels = new Hashtable<Integer, JLabel>();
		labels.put(0, createLabel("0", null));
		labels.put(5, createLabel("5", null));
		labels.put(10, createLabel("10", null));
		labels.put(15, createLabel("15", null));
		dcoAftertouchSens = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "DCO ATOUCH SENS", STD_SLIDER, DCO,  16, JSlider.HORIZONTAL, 15, 13, labels);
		labels = new Hashtable<Integer, JLabel>();
		labels.put(0, createLabel("00", null));
		labels.put(1, createLabel("01", null));
		labels.put(2, createLabel("02", null));
		labels.put(3, createLabel("03", null));
		dcoPulseWaveform = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "DCO PULSE WF", STD_SLIDER, DCO, 4, JSlider.HORIZONTAL, 3, 3, labels);
		labels = new Hashtable<Integer, JLabel>();
		labels.put(0, createLabel("00", null));
		labels.put(1, createLabel("01", null));
		labels.put(2, createLabel("02", null));
		labels.put(3, createLabel("03", null));
		labels.put(4, createLabel("04", null));
		labels.put(5, createLabel("05", null));
		dcoSawtoothWaveform = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "DCO SAW WF", STD_SLIDER, DCO, 6, JSlider.HORIZONTAL, 5, 4, labels);
		labels = new Hashtable<Integer, JLabel>();
		labels.put(0, createLabel("00", null));
		labels.put(1, createLabel("01", null));
		labels.put(2, createLabel("02", null));
		labels.put(3, createLabel("03", null));
		labels.put(4, createLabel("04", null));
		labels.put(5, createLabel("05", null));
		dcoSubOSCWaveform = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "DCO SUBOSC WF", STD_SLIDER, DCO, 6, JSlider.HORIZONTAL, 5, 5, labels);
		dcoSubOSCLevel = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "DCO SUBOSC LVL", STD_SLIDER, DCO, 4, JSlider.HORIZONTAL, 3, 7, labels );
		
		labels = new Hashtable<Integer, JLabel>();
		labels.put(0, createLabel("0", null));
		labels.put(1, createLabel("1", null));
		labels.put(2, createLabel("2", null));
		labels.put(3, createLabel("3", null));
		dcoNoiseLevel = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "DCO NOISE LVL", STD_SLIDER, DCO, 4, JSlider.HORIZONTAL, 3, 8, labels);
		dcoPWMDepth = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "DCO PWM DEPTH", STD_SLIDER, DCO, 128, JSlider.HORIZONTAL, 63, 14, null);
		
		vcaLevel = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "VCA LEVEL", STD_SLIDER, VCA, 128, JSlider.HORIZONTAL, 63, 22, null);	
		labels = new Hashtable<Integer, JLabel>();
		labels.put(0, createLabel("ENV", null));
		labels.put(1, createLabel("Gate", null));
		labels.put(2, createLabel("ENV w/ Dyn", null));
		labels.put(3, createLabel("Gate w/ Dyn", null));
		vcaENVMode = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "VCA ENV MODE", STD_SLIDER, VCA, 4, JSlider.VERTICAL, 3, 2, labels);
		labels = new Hashtable<Integer, JLabel>();
		labels.put(0, createLabel("0", null));
		labels.put(5, createLabel("5", null));
		labels.put(10, createLabel("10", null));
		labels.put(15, createLabel("15", null));
		vcaAftertouchSens = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "VCA ATOUCH SENS", STD_SLIDER, VCA, 16, JSlider.HORIZONTAL, 15, 23, labels);
		
		
		// CHORUS BLOCK
		labels = new Hashtable<Integer, JLabel>();
		labels.put(0, createLabel("OFF", null));
		labels.put(1, createLabel("ON", null));
		chorus = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "CHORUS", STD_SLIDER, CHORUS, 2, JSlider.VERTICAL, 1, 10, labels);
		chorusRate = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "CHORUS RATE", STD_SLIDER, CHORUS, 128, JSlider.HORIZONTAL, 63, 34, null);
		
		labels = new Hashtable<Integer, JLabel>();
		labels.put(0, createLabel("0", null));
		labels.put(1, createLabel("1", null));
		labels.put(2, createLabel("2", null));
		labels.put(3, createLabel("3", null));
		hpfCutoffFreq = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "HPF CUTOFF FREQ", STD_SLIDER, HPF, 4, JSlider.HORIZONTAL, 3, 9, labels);
		
		envT1 = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "ENV T1", STD_SLIDER, ENV, 128, JSlider.VERTICAL, 60, 26, null);
		envT2 = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "ENV T2", STD_SLIDER, ENV, 128, JSlider.VERTICAL, 91, 28, null);
		envT3 = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "ENV T3", STD_SLIDER, ENV, 128, JSlider.VERTICAL, 106, 30, null);
		envT4 = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "ENV T4", STD_SLIDER, ENV, 128, JSlider.VERTICAL, 74, 32, null);
		envL1 = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "ENV L1", STD_SLIDER, ENV, 128, JSlider.VERTICAL, 103, 27, null);
		envL2 = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "ENV L2", STD_SLIDER, ENV, 128, JSlider.VERTICAL, 127, 29, null);
		envL3 = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "ENV L3", STD_SLIDER, ENV, 128, JSlider.VERTICAL, 54, 31, null);
		labels = new Hashtable<Integer, JLabel>();
		labels.put(0, createLabel("0", null));
		labels.put(5, createLabel("5", null));
		labels.put(10, createLabel("10", null));
		labels.put(15, createLabel("15", null));
		envKeyFollow = new MIDIModulator(controllersPanel, envWidget, signalsViewport,  "ENV KEY FOLLOW", STD_SLIDER, ENV, 16, JSlider.HORIZONTAL, 15, 33, labels);
		
		
		ModulatorsState.setDefaultLocations();
		envWidget.initialize();
	}
	
	public static void main(String[] args) throws Exception 
	{  
		try 
		{	
		    for (LookAndFeelInfo info : UIManager.getInstalledLookAndFeels()) 
		    {
		        if ("Nimbus".equals(info.getName())) 
		        {
		            UIManager.setLookAndFeel(info.getClassName());
		            break;
		        }
		    }
		} 
		catch (Exception e) 
		{
			// If Nimbus is not available, you can set the GUI to another look and feel.
		}
		
		EventQueue.invokeLater(new Runnable()
		{
			@Override
			public void run()
			{
				try 
				{
					new ILoveMyJuno2();
				} 
				catch (Exception e) 
				{
				}
			}
		});
	}

	public void actionPerformed(ActionEvent e) 
	{
		String actionCommand = e.getActionCommand();
		if (e.getSource() instanceof JMenuItem)
		{
			if ("Quit".equals(actionCommand))
			{
				ModulatorsState.storeLocations();
				frame.dispose();
				System.exit(0);
			}
			/*
			else if ("Open Tone Bank".equals(actionCommand))
			{
				JFileChooser fileChooser = new JFileChooser();
				fileChooser.setFont(new Font("Arial", Font.PLAIN, 22));
			    FileNameExtensionFilter filter = new FileNameExtensionFilter(
			        "Roland aJuno 1/2 System Exclusive Bulk Dump File (*.syx, *.sysex)", "syx", "sysex");
			    fileChooser.setFileFilter(filter);
			    int returnVal = fileChooser.showOpenDialog(frame);
			    if(returnVal == JFileChooser.APPROVE_OPTION)
			    {
			    	File file = fileChooser.getSelectedFile();
			    	byte[] dump = FileIO.loadBulkDump( file );
			    	// Check whether the dump is valid
			    	if (MIDIMessageDecoder.decodeDump( dump ))
					{
						// The dump is valid; put it as the last entry into the librarian
			    		librarian.addToneBank( dump);
					}
			    	else
			    	{
			    		// The dump is corrupt or is not of the desired format
			    		System.out.println("The Bulk Dump is corrupt - could not save it");
			    	}
			    }
			    else if (returnVal == JFileChooser.ERROR_OPTION)
			    {
			    	
			    	System.out.println("An error occurred while opening the file.");
			    }
					
			}
			*/
			else if ("Reserve MIDI Ports".equals(actionCommand))
			{
				mpController.setLocation(200, 200);
				mpController.setVisible(true);
			}	
			else if ("Use Remote Keyboard".equals(actionCommand))
			{
				mkController.setLocation(200, 200);
				mkController.setVisible(true);
			}
			else if ("Display Block Diagram".equals(actionCommand))
			{
				exchangeViewport();
			}
			else if ("Jump to Librarian".equals(actionCommand))
			{
				JViewport currentViewport = controllersScrollPane.getViewport();
				HashSet<JComponent> components = controllersPanel.getAlwaysVisibleComponents();
				Iterator<JComponent> it = components.iterator();
				while (it.hasNext())
				{
					JComponent c = (JComponent) it.next();
					if (c instanceof LibrarianWidget)
					{
						if (currentViewport instanceof BlockDiagramViewport) exchangeViewport();
						signalsViewport.setViewPosition(c.getLocation());
						break;
					}
				}
			}
			else if ("Jump to Envelope".equals(actionCommand))
			{
				JViewport currentViewport = controllersScrollPane.getViewport();
				HashSet<JComponent> components = controllersPanel.getAlwaysVisibleComponents();
				Iterator<JComponent> it = components.iterator();
				while (it.hasNext())
				{
					JComponent c = (JComponent) it.next();
					if (c instanceof EnvelopeWidget)
					{
						if (currentViewport instanceof BlockDiagramViewport) exchangeViewport();
						signalsViewport.setViewPosition(c.getLocation());
						break;
						
					}
				}
			}
		}
		else if (e.getSource() == holdKey)
		{
			Piano.setHoldNote(holdKey.isSelected());
			MIDIPlugsModel.sendAllNotesOff();
			
		}
		else if (e.getSource() == allNotesOff)
		{
			MIDIPlugsModel.sendAllNotesOff();
		}
		else if (e.getSource() == transposeDown)
		{
			if (!transposeDown.isSelected())
			{
				transposeDown.setSelected(false);
				dontTranspose.setSelected(true);
				piano.setTransposition(24);
			}
			else
			{
				transposeDown.setSelected(true);
				transposeUp.setSelected(false);
				dontTranspose.setSelected(false);
				piano.setTransposition(0);
			}
		}
		else if (e.getSource() == transposeUp)
		{
			if (!transposeUp.isSelected())
			{
				transposeUp.setSelected(false);
				dontTranspose.setSelected(true);
				piano.setTransposition(24);
			}
			else
			{
				transposeUp.setSelected(true);
				transposeDown.setSelected(false);
				dontTranspose.setSelected(false);
				piano.setTransposition(36);
			}
		}
		/*
		else if ("Receive Tone Bank".equals(actionCommand))
		{
			if (receiveDialog != null) 
			{
				System.out.println("BulkReceiveDialog-Instanz existierte und wurde gelöscht.");
				receiveDialog = null;
			}
			if (receiveDialog != null) { System.out.println("Eine BulkReceiveDialog-Instanz besteht nach wie vor."); }
			receiveDialog = MIDIBulkReceiveDialog.getInstance();
			receiveDialog.setMIDIPlugsModel(mpModel);
			receiveDialog.listen();
			receiveDialog.setLocation(200, 200);
			receiveDialog.setVisible(true);
		}
		else if ("Transmit Tone Bank".equals(actionCommand))
		{
			dumpDialog = MIDIBulkDumpDialog.getInstance();
			dumpDialog.setMIDIPlugsModel(mpModel);
			dumpDialog.setLocation(200, 200);
			dumpDialog.setVisible(true);
			
		}
		else if ("Open Tone Bank".equals(actionCommand))
		{
			JFileChooser fileChooser = new JFileChooser();
			fileChooser.setFont(new Font("Arial", Font.PLAIN, 22));
		    FileNameExtensionFilter filter = new FileNameExtensionFilter(
		        "Roland aJuno 1/2 SysEx Bulk Dump File (*.syx, *.sysex)", "syx", "sysex");
		    fileChooser.setFileFilter(filter);
		    int returnVal = fileChooser.showOpenDialog(frame);
		    if(returnVal == JFileChooser.APPROVE_OPTION)
		    {
		    	File file = fileChooser.getSelectedFile();
		    	byte[] dump = FileIO.loadBulkDump( file );
		    	// Check whether the dump is valid
		    	if (MIDIMessageDecoder.decodeDump( dump ))
				{
					// The dump is valid; put it as the last entry into the librarian
		    		librarian.addToneBank( dump );
				}
		    	else
		    	{
		    		// The dump is corrupt or is not of the desired format
		    		System.out.println("The Bulk Dump is corrupt - could not save it");
		    	}
		    }
		}
		*/
	}
	
	
	private JLabel createLabel(String name, Icon img)
	{
		JLabel label = null;
		if (name != null)
		{
			label = new JLabel(name);
			label.setForeground(new Color(190, 190, 225));
			label.setFont(new Font("Arial", Font.PLAIN, 22));
		}
		else if (img != null)
		{
			label = new JLabel(img);
			
		}
		return label;
	}

	@Override
	public void componentHidden(ComponentEvent arg0) {
		// TODO Auto-generated method stub
		
	}

	@Override
	public void componentMoved(ComponentEvent arg0) {
		// TODO Auto-generated method stub
		
	}

	@Override
	public void componentResized(ComponentEvent e) 
	{
		Insets frameInsets = frame.getInsets();
		pianoScrollPane.setPreferredSize(new Dimension(frame.getWidth()-frameInsets.left-frameInsets.right, 184));
		pianoScrollPane.setMinimumSize(new Dimension(frame.getWidth()-frameInsets.left-frameInsets.right, 184));
		pianoScrollPane.revalidate();
		controllersScrollPane.setSize(new Dimension(frame.getWidth()-frameInsets.left-frameInsets.right, frame.getHeight()-250));
		controllersScrollPane.revalidate();
	}

	@Override
	public void componentShown(ComponentEvent arg0) 
	{
	}

	public static MIDIPlugsModel getMIDIPlugsModel() { return mpModel; }
	
	
	public void exchangeViewport()
	{
		JViewport currentViewport = controllersScrollPane.getViewport();
		if (currentViewport instanceof SignalsViewport)
		{
			if (blockDiagramViewport == null) 
			{
				blockDiagramViewport = new BlockDiagramViewport(controllersPanel, this);
			}
			controllersScrollPane.setViewport(blockDiagramViewport);
			controllersScrollPane.getVerticalScrollBar().setCursor(new Cursor(Cursor.HAND_CURSOR));
			controllersScrollPane.getHorizontalScrollBar().setCursor(new Cursor(Cursor.HAND_CURSOR));
			controllersScrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
			controllersScrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
			blockDiagramViewport.setPreferredSize(blockDiagramViewport.getView().getSize());
			blockDiagramViewport.setSize(blockDiagramViewport.getView().getSize());
			blockDiagramViewport.revalidate();
		}
		else
		{
			controllersScrollPane.setViewport(signalsViewport);
			controllersScrollPane.getVerticalScrollBar().setCursor(new Cursor(Cursor.HAND_CURSOR));
			controllersScrollPane.getHorizontalScrollBar().setCursor(new Cursor(Cursor.HAND_CURSOR));
			controllersScrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
			controllersScrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
			signalsViewport.setPreferredSize(signalsViewport.getView().getSize());
			signalsViewport.setSize(signalsViewport.getView().getSize());
			signalsViewport.revalidate();
		}
		controllersScrollPane.getVerticalScrollBar().setUnitIncrement(10);
		controllersScrollPane.getHorizontalScrollBar().setUnitIncrement(10);
	}
}


/*
class MyScrollBarUI extends javax.swing.plaf.synth.SynthScrollBarUI
{
	//protected JButton createIncreaseButton(int orientation) {};
		 
		//protected JButton createDecreaseButton(int orientation)
		 
	protected void configureScrollBarColors()
	{
		
		thumbColor = new Color(200, 0, 0);
		thumbHighlightColor = new Color(150, 150, 190);
		thumbLightShadowColor = new Color(100, 150, 190);
		thumbDarkShadowColor = new Color(100, 150, 190);
		trackColor = new Color(100, 250, 100);
		trackHighlightColor = new Color(200, 200, 250);
		
	}
}
*/

/*
class MyListCellRenderer extends DefaultListCellRenderer
{
    public Component getListCellRendererComponent(JList list,
                                              Object value,
                                              int index,
                                              boolean isSelected,
                                              boolean cellHasFocus)
    {
        Component c = super.getListCellRendererComponent(
            list,value,index,isSelected,cellHasFocus);
        c.setForeground(index==0 ? Color.red : Color:black);
    }
} 
*/