package gui;

import java.awt.BorderLayout;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.EventQueue;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.util.Iterator;
import java.util.Vector;

import javax.sound.midi.MidiDevice;
import javax.sound.midi.MidiMessage;
import javax.sound.midi.MidiUnavailableException;
import javax.sound.midi.Receiver;
import javax.sound.midi.Transmitter;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextArea;
import javax.swing.SwingConstants;
import javax.swing.WindowConstants;

import utils.DumpReceiver;
import utils.FileIO;
import utils.MIDIBulk;
import utils.MIDIMessageAndTimestamp;
import utils.MIDIMessageDecoder;

import main.Constants;
import main.MIDIPlugsModel;

// Singleton class that realizes a MIDI Receive dialog
public class MIDIBulkReceiveDialog extends JDialog implements ActionListener, WindowListener, Constants
{
	private static final long serialVersionUID = 1L;
	JTextArea textArea = new JTextArea();
	static JButton store = new JButton("Store");
	static JButton discard = new JButton("Discard");
	Thread listener;
	Receiver byteReceiver;
	Transmitter transmitter;
	private static MIDIPlugsModel mpModel;
	public static Vector<MIDIMessageAndTimestamp> receivedMessages = new Vector<MIDIMessageAndTimestamp>();
	private static JLabel messageLabel = new JLabel("Waiting for transmission", SwingConstants.CENTER);
	private static int state = BORED;
	private static MIDIBulkReceiveDialog instance = null;
	
	
	public static MIDIBulkReceiveDialog getInstance() 
	{
        if (instance == null) 
        {
            instance = new MIDIBulkReceiveDialog();
        }
        return instance;
    }

	
	
	private MIDIBulkReceiveDialog()
	{
		//this.setModal(true); // AVOID: Otherwise, the MIDI transmission fails!
		this.setTitle("Receive Tone Bank from aJuno");
		this.setLayout(new BorderLayout());
		this.setResizable(false);
		this.setAlwaysOnTop(true);
		this.requestFocus();
		this.addWindowListener(this);
		this.setFocusTraversalKeysEnabled(false);
		this.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
		this.setCursor(new Cursor(Cursor.HAND_CURSOR));
		this.setPreferredSize(new Dimension(500, 300));
		this.pack();
		this.setVisible(false);
		textArea.setText("Hold 'Data Transfer', 'Write' and 'Bulk Receive' on your Juno. The display should read 'Bulk Receive MIDI.o'.");
		textArea.setFont(new Font("Arial", Font.PLAIN, 22));
		textArea.setLineWrap(true);
		
		messageLabel.setFont(new Font("Arial", Font.PLAIN, 18));
		messageLabel.setPreferredSize(new Dimension(250, 30));
		
		this.add(textArea, BorderLayout.NORTH);
		this.add(messageLabel, BorderLayout.CENTER);
		store.setFont(new Font("Arial", Font.PLAIN, 18));
		store.setEnabled(false);
		store.addActionListener(this);
		
		discard.setFont(new Font("Arial", Font.PLAIN, 18));
		discard.addActionListener(this);
		
		JPanel lowerPanel = new JPanel();
		lowerPanel.add(store);
		lowerPanel.add(discard);
		
		this.add(lowerPanel, BorderLayout.SOUTH);
		this.setVisible(true);
	}
	
	
	@Override
	public void actionPerformed(ActionEvent e) 
	{
		// Discard button was clicked
		if (e.getSource() == discard)
		{
			MIDIMessageDecoder.deleteLastDump();
			exit();
		}
		else if (e.getSource() == store)
		{
			boolean sane = MIDIMessageDecoder.decodeLastDump();
			if (sane) MIDIMessageDecoder.addLastDumpToLibrarian();
			exit();
		}
	}
	
	
	private class Listener implements Runnable
	{
		public Listener()
		{
			MidiDevice inDevice = mpModel.getReservedInDevice();
			if ((inDevice != null) && (inDevice.isOpen()))
			{
				byteReceiver = new DumpReceiver(System.out);
				try 
				{
					transmitter = inDevice.getTransmitter();
					transmitter.setReceiver(byteReceiver);
				} 
				catch (MidiUnavailableException e) 
				{
					inDevice.close();
				}
			}
		}
		
		@Override
		public void run()
		{
			while (true)
			{
				// Do nothing
			}
		}

		private Vector<MIDIMessageAndTimestamp> cloneVector( Vector<MIDIMessageAndTimestamp> receivedMessages) 
		{
			if (receivedMessages.isEmpty()) return null;
			Vector<MIDIMessageAndTimestamp> v = new Vector<MIDIMessageAndTimestamp>( receivedMessages.size());
			Iterator<MIDIMessageAndTimestamp> it = receivedMessages.iterator();
			while (it.hasNext())
			{
				MIDIMessageAndTimestamp mat = (MIDIMessageAndTimestamp) it.next();
				MIDIMessageAndTimestamp newMat = new MIDIMessageAndTimestamp( mat );
				v.add(newMat);
			}
			
			receivedMessages.clear();
			return v;
		}
	}


	
	public static void addMidiMessage(MIDIMessageAndTimestamp midiMessageAndTimestamp) 
	{
		receivedMessages.add(midiMessageAndTimestamp);
		int num = receivedMessages.size();
		messageLabel.setText(""+num+" messages received");

		if (num == 16)
		{
			messageLabel.setText("Waiting for transmission");
			if ((receivedMessages != null) && (!receivedMessages.isEmpty()))
			{
				MIDIMessageDecoder.addReceivedDump(receivedMessages);
				receivedMessages.clear();
			}
			
			// All messages have been received - the bulk dump may be stored
			store.setEnabled(true);
			
		}
	}
	
	
	public void setMIDIPlugsModel(MIDIPlugsModel model)
	{
		this.mpModel = model;
	}
	
	
	public void listen()
	{
		// Immediately start listening to the MIDI IN device
		// to which the synthesizer is attached
		listener = new Thread(new Listener());
		listener.start();
	}
	
	
	private void exit()
	{
		this.setVisible(false);
		
		if (transmitter != null) { transmitter.close(); transmitter = null; }
		if (byteReceiver != null) { byteReceiver.close(); byteReceiver = null; }
	}
	
	
	
	public void windowClosed(WindowEvent arg0) 
	{
		exit();
	}
	@Override
	public void windowActivated(WindowEvent arg0) {}
	@Override
	public void windowClosing(WindowEvent arg0) {}
	@Override
	public void windowDeactivated(WindowEvent arg0) {}
	@Override
	public void windowDeiconified(WindowEvent arg0) {}
	@Override
	public void windowIconified(WindowEvent arg0) {}
	@Override
	public void windowOpened(WindowEvent arg0) {}
	
	
	public static synchronized void setState(int newState) { state = newState; }

}
